/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.client;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.actions.AggregationDetails;
import org.jboss.soa.esb.actions.Aggregator;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.registry.RegistryException;

import java.util.*;

/**
 * Message Multicaster.
 * <p/>
 * Used to send a message to a recipient list, or a subset of that recipient list.
 * <p/>
 * Caches a {@link ServiceInvoker} instance for each recipient.
 * <p/>
 * <b>Note</b>:  This doesn't "multicast" message delivery in the true IP sense of the
 * word "multicast".  It sends the message to all recipients in its recipient list,
 * one at a time, in the order the recipients were added via the
 * {@link #addRecipient(org.jboss.soa.esb.Service)} method.
 *
 * @author <a href="mailto:tom.fennely@jboss.com">tom.fennelly@jboss.com</a>
 */
public class MessageMulticaster {

    private static Logger logger = Logger.getLogger(MessageMulticaster.class);

    private Map<Service, ServiceInvoker> invokers = new LinkedHashMap<Service, ServiceInvoker>();
    private String splitId;
    private boolean aggregatorOnProperties = false; // By default... set on context.

    /**
     * Public default constructor.
     */
    public MessageMulticaster() {        
    }

    /**
     * Public default constructor.
     * @param splitId Split ID for this multicaster.
     */
    public MessageMulticaster(String splitId) {
        AssertArgument.isNotNullAndNotEmpty(splitId, "splitId");
        this.splitId = splitId;
    }

    /**
     * Add a message recipient Service.
     * @param service Recipient service for receipt of messages from this miltcaster instance.
     * @throws RegistryException Failed to lookup Service endpoint.
     * @throws MessageDeliverException Failed to deliver message to endpoint.
     */
    public void addRecipient(Service service) throws RegistryException, MessageDeliverException {
        AssertArgument.isNotNull(service, "service");
        // The ServiceInvoker will be lazilly created in the getInvoker method.
        // This is effectively a registration of the service as being a potential recipient.
        invokers.put(service, null);
    }

    /**
     * Is the specified service a recipient of this multicaster instance.
     * @param service The service to check for.
     * @return True if the supplied service is one of the recipients, otherwise false.
     */
    public boolean isRecipient(Service service) {
        return invokers.containsKey(service);
    }

    /**
     * Get the number ot recipients associated with this multicaster instance.
     * @return The number of recipients.
     */
    public int getRecipientCount() {
        return invokers.size();
    }

    /**
     * Send the message to all the recipients associated with this multicaster.
     * @param message The message.
     * @throws RegistryException Failed to lookup Service endpoint.
     * @throws MessageDeliverException Failed to deliver message to endpoint.
     */
    public void sendToAll(Message message) throws RegistryException, MessageDeliverException {
        sendToSubset(message, new ArrayList<Service>(invokers.keySet()));
    }

    /**
     * Send the message to all the recipients associated with this multicaster and
     * also listed in the supplied recipient list.
     * <p/>
     * The recipients supplied in the list must have been added through the {@link #addRecipient(org.jboss.soa.esb.Service)}
     * method, otherwise the message will be delivered to the Dead Letter Channel.
     *
     * @param message The message.
     * @param recipients The recipient subset to which the supplied message is to be sent.
     * @throws RegistryException Failed to lookup Service endpoint.
     * @throws MessageDeliverException Failed to deliver message to endpoint.
     */
    public void sendToSubset(Message message, List<Service> recipients) throws RegistryException, MessageDeliverException {
        if (recipients.isEmpty()) {
            logger.warn("MessageMulticaster.sendToSubset: empty recipients list!");
        } else{
            String seriesUUID = UUID.randomUUID().toString();
            long seriesTimestamp = System.currentTimeMillis();
            int recipientCount = recipients.size();

            for(int i = 0; i < recipientCount; i++) {
                Service recipient = recipients.get(i);
                ServiceInvoker invoker = getInvoker(recipient);

                if(recipientCount > 1) {
                    // Only add aggregation info if we're splitting the message i.e. sending it to
                    // more than 1 recipient...
                    addAggregationDetails(message, seriesUUID, recipientCount, seriesTimestamp, i + 1);
                }

                if(invoker == null) {
                    logger.error("Service '" + recipient + "' is not in recipient list.  Delivering message to Dead Letter Channel.");
                    ServiceInvoker.deliverToDeadLetterService(message);
                } else {
                    try {
                        invoker.deliverAsync(message);
                    } catch(MessageDeliverException e) {
                        logger.error("Failed to deliver message to Service '" + recipient + "'.  Delivering message to Dead Letter Channel.");
                        ServiceInvoker.deliverToDeadLetterService(message);
                    }
                }
            }
        }
    }

    protected void addAggregationDetails(Message message, String uuId, int recipientCount, long seriesTimestamp, int messageIndex) {
        AggregationDetails aggrDetails = new AggregationDetails(uuId, messageIndex, recipientCount, seriesTimestamp);
        ArrayList<String> aggregatorTags;
        
        if(aggregatorOnProperties) {
        	aggregatorTags = (ArrayList<String>) message.getProperties().getProperty(Aggregator.AGGEGRATOR_TAG);
        } else {
        	aggregatorTags = (ArrayList<String>) message.getContext().getContext(Aggregator.AGGEGRATOR_TAG);
        }

        // This is useful during aggregation - as a way of id'ing where the split occurred.
        aggrDetails.setSplitId(splitId);

        if ((aggregatorTags == null) || !aggregatorOnProperties) {
            if (aggregatorTags == null) {
                aggregatorTags = new ArrayList<String>();
            } else {
                aggregatorTags = new ArrayList<String>(aggregatorTags) ;
            }
            
            if(aggregatorOnProperties) {
            	message.getProperties().setProperty(Aggregator.AGGEGRATOR_TAG, aggregatorTags);
            } else {
            	message.getContext().setContext(Aggregator.AGGEGRATOR_TAG, aggregatorTags);
            }
        }

        if(messageIndex > 1) {
            // remove the tag string for the last recipient...
            aggregatorTags.remove(aggregatorTags.size() - 1);
        }
        aggregatorTags.add(aggrDetails.toString());

        if (logger.isDebugEnabled()) {
            logger.debug(Aggregator. AGGEGRATOR_TAG + "=" + aggrDetails);
        }
    }

	public void setAggregatorOnProperties(boolean aggregatorOnProperties) {
		this.aggregatorOnProperties = aggregatorOnProperties;
	}

	private ServiceInvoker getInvoker(Service recipient) throws RegistryException, MessageDeliverException {
        ServiceInvoker invoker = invokers.get(recipient);

        // We lazilly create the invokers...
        if(invoker == null) {
            if(!invokers.containsKey(recipient)) {
                // We don't create an invoker for the Service if it wasn't
                // already "registered" via the addRecipient method.
                return null;
            }
            invoker = new ServiceInvoker(recipient);
            invokers.put(recipient, invoker);
        }

        return invoker;
    }
}