/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;

import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.IllegalBlockSizeException;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.common.Configuration;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.services.security.util.CryptoUtil;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;

/**
 * Util for encrypting/decrypting using assymmetric keys.
 *
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public enum PublicCryptoUtil
{
    INSTANCE;

    private static final Logger logger = Logger.getLogger(PublicCryptoUtil.class);

    private Logger log;
    private Key key;
    private PublicKey publicKey;
    private String transformation;
    private boolean isSecurityConfigured;

    private PublicCryptoUtil()
    {
        try
        {
            log = getLogger();
            init();
        }
        catch (final Exception e)
        {
            throw new IllegalStateException("Unknown algorithm:", e);
        }
    }

    public byte[] encrypt(final Serializable object) throws SecurityServiceException
    {
        if (!isSecurityConfigured)
        {
            return null;
        }
    
        final ByteArrayOutputStream encryptedOutStream = new ByteArrayOutputStream();
        final ByteArrayInputStream plainInStream;
        try
        {
            plainInStream = new ByteArrayInputStream(getBytes(object));
        }
        catch (final IOException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
    
        try
        {
            byte[] buf = new byte[100];
            int bufLength;
            final Cipher cipher = CryptoUtil.getCipherForEncryption(transformation, publicKey);
            while ( (bufLength = plainInStream.read(buf)) != -1)
            {
                byte[] tmp = cipher.doFinal(copyBytes(buf, bufLength));
                encryptedOutStream.write(tmp);
            }
            encryptedOutStream.flush();
            return encryptedOutStream.toByteArray();
        }
        catch (final IOException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (IllegalBlockSizeException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (BadPaddingException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
    }

    public Serializable decrypt(final byte[] bytes) throws SecurityServiceException
    {
        if (!isSecurityConfigured)
        {
            return null;
        }
        final ByteArrayInputStream encryptedBytesInStream = new ByteArrayInputStream(bytes);
        final ByteArrayOutputStream decryptedBytesOutStream = new ByteArrayOutputStream();
    
        byte[] decryptBytes = null;
        try
        {
            byte[] buf = new byte[128];
            int bufLength;
            final Cipher cipher = CryptoUtil.getCipherForDecryption(transformation, key);
            while ( (bufLength = encryptedBytesInStream.read(buf)) != -1)
            {
                byte[] tmp = cipher.doFinal(copyBytes(buf, bufLength));
                decryptedBytesOutStream.write(tmp);
            }
            decryptedBytesOutStream.flush();
            decryptBytes = decryptedBytesOutStream.toByteArray();
        }
        catch (final IOException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (IllegalBlockSizeException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (BadPaddingException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        finally
        {
            try { decryptedBytesOutStream.close(); } catch (IOException ignore) { log.error(ignore.getMessage(),ignore); }
        }
    
        return toSerializable(decryptBytes);
    }

    private void init() throws SecurityServiceException
    {
        String keystorePath = Configuration.getSecurityServicePublicKeystore();
        if (keystorePath == null)
        {
            log.info("No public keystore was specified in jbossesb-properites.xml. Add '" + Environment.SECURITY_SERVICE_PUBLIC_KEYSTORE + "'");
        }
        else
        {
            isSecurityConfigured = true;
            try
            {
                String keystoreType = Configuration.getSecurityServicePublicKeystoreType();
                if (keystoreType == null)
                {
                    keystoreType = KeyStore.getDefaultType();
                }
                
                String keystorePassword = Configuration.getSecurityServicePublicKeystorePassword();
                String privateKeyPass = Configuration.getSecurityServicePublicKeyPassword();
                String privateKeyAlias = Configuration.getSecurityServicePublicKeyAlias();
                
                // Try to retreive the password from a file if applicable.
                keystorePassword = getPasswordFromFile(keystorePassword);
                privateKeyPass = getPasswordFromFile(privateKeyPass);

                try
                {
                    KeyStore keystore = KeyStore.getInstance(keystoreType);
                    InputStream in = ClassUtil.getResourceAsStream(keystorePath, this.getClass());
                    if (in == null)
                    {
                        throw new SecurityServiceException("Could not locate public keystore using '" + keystorePath + "'");
                    }

                    // load the keystore contents
                    keystore.load(in, keystorePassword.toCharArray());
                    key = keystore.getKey(privateKeyAlias, privateKeyPass.toCharArray());
                    Certificate certificate = keystore.getCertificate(privateKeyAlias);
                    publicKey = certificate.getPublicKey();

                    // "algorithm/mode/padding" or defaults to "algorithm"
                    transformation = Configuration.getSecurityServicePublicKeyTransformation();
                    if (transformation == null)
                    {
                        this.transformation = key.getAlgorithm();
                    }
                }
                finally
                {
                    keystorePassword = null;
                    privateKeyAlias = null;
                    privateKeyPass = null;
                }
            }
            catch (final KeyStoreException e)
            {
                throw new SecurityServiceException(e.getMessage(), e);
            }
            catch (final NoSuchAlgorithmException e)
            {
                throw new SecurityServiceException(e.getMessage(), e);
            }
            catch (final CertificateException e)
            {
                throw new SecurityServiceException(e.getMessage(), e);
            }
            catch (final IOException e)
            {
                throw new SecurityServiceException(e.getMessage(), e);
            }
            catch (final UnrecoverableKeyException e)
            {
                throw new SecurityServiceException(e.getMessage(), e);
            }
        }
    }
    
    private String getPasswordFromFile(final String password) 
    {
        if (PasswordUtil.isPasswordFile(password))
        {
            try
            {
                return new PasswordUtil(password).getPasswordAsString();
            }
            catch (final IOException e)
            {
                throw new IllegalStateException(e.getMessage(), e);
            }
        }
        
        return password;
    }

    private static byte[] getBytes(final Serializable ser) throws IOException
    {
        ByteArrayOutputStream bout = new ByteArrayOutputStream();
        ObjectOutputStream oout = new ObjectOutputStream(bout);
        oout.writeObject(ser);
        return bout.toByteArray();
    }

    private Serializable toSerializable(final byte[] decryptBytes) throws SecurityServiceException
    {
        ObjectInputStream inputStream = null;
        try
        {
            inputStream = new ObjectInputStream(new ByteArrayInputStream(decryptBytes));
            return (Serializable) inputStream.readObject();
        }
        catch (final IOException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        catch (final ClassNotFoundException e)
        {
            throw new SecurityServiceException(e.getMessage(), e);
        }
        finally
        {
            try { inputStream.close(); } catch (IOException ignore) { log.error(ignore.getMessage(), ignore); }
        }
    }

    private static byte[] copyBytes(byte[] bytes, int length)
    {
        if (bytes.length == length)
        {
            return bytes;
        }

        byte[] newBytes = new byte[length];
        for (int i = 0; i < length; i++)
        {
            newBytes[i] = bytes[i];
        }
        return newBytes;
    }

    private static Logger getLogger()
    {
        return Logger.getLogger(PublicCryptoUtil.class);
    }

    public boolean addAuthRequestToMessage(AuthenticationRequest authRequest, Message message) throws MessageDeliverException {
        if(authRequest != null) {
            try {
                byte[] encrypted = PublicCryptoUtil.INSTANCE.encrypt((Serializable) authRequest);
                if (encrypted != null) {
                    message.getContext().setContext(SecurityService.AUTH_REQUEST, encrypted);
                    return true;
                }
                logger.warn("No public keystore has been configured which means that the authentication request cannot be encrypted. Please configure jbossesb-properties.xml with a publickey store.");
            } catch (final SecurityServiceException e) {
                throw new MessageDeliverException(e.getMessage(), e);
            }
        }

        return false;
    }

    public boolean isAuthRequestOnMessage(Message message) {
        return (message.getContext().getContext(SecurityService.AUTH_REQUEST) != null);
    }
}
