/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.util;

import java.util.zip.ZipOutputStream;
import java.util.zip.ZipEntry;
import java.util.Map;
import java.util.LinkedHashMap;
import java.util.Set;
import java.util.jar.JarInputStream;
import java.io.*;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.apache.log4j.Logger;

/**
 * Deployment archive.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class DeploymentArchive {

    private static Logger logger = Logger.getLogger(DeploymentArchive.class);
    private String archiveName;
    private LinkedHashMap<String, byte[]> entries = new LinkedHashMap<String, byte[]>();

    /**
     * Public constructor.
     * @param archiveName The archive name of the deployment.
     */
    public DeploymentArchive(String archiveName) {
        AssertArgument.isNotNull(archiveName, "archiveName");
        this.archiveName = archiveName;
    }

    /**
     * Get the name of the deployment associated with this archive.
     * @return The name of the deployment.
     */
    public String getArchiveName() {
        return archiveName;
    }

    /**
     * Add the supplied data as an entry in the deployment.
     *
     * @param path The target path of the entry when added to the archive.
     * @param data The data.
     */
    public void addEntry(String path, InputStream data) {
        AssertArgument.isNotNullAndNotEmpty(path, "path");
        AssertArgument.isNotNull(data, "data");

        try {
            entries.put(trimLeadingSlash(path.trim()), StreamUtils.readStream(data));
        } finally {
            try {
                data.close();
            } catch (IOException e) {
                logger.warn("Unable to close input stream for archive entry '" + path + "'.");
            }
        }
    }

    /**
     * Add the supplied data as an entry in the deployment.
     *
     * @param path The target path of the entry when added to the archive.
     * @param data The data.
     */
    public void addEntry(String path, byte[] data) {
        AssertArgument.isNotNullAndNotEmpty(path, "path");
        AssertArgument.isNotNull(data, "data");

        entries.put(trimLeadingSlash(path.trim()), data);
    }

    /**
     * Add an "empty" entry in the deployment.
     * <p/>
     * Equivalent to adding an empty folder.
     *
     * @param path The target path of the entry when added to the archive.
     */
    public void addEntry(String path) {
        AssertArgument.isNotNullAndNotEmpty(path, "path");

        path = path.trim();
        if(path.endsWith("/")) {
            entries.put(trimLeadingSlash(path), null);
        } else {
            entries.put(trimLeadingSlash(path) + "/", null);
        }
    }

    /**
     * Add the specified classpath resource as an entry in the deployment.
     *
     * @param path The target path of the entry when added to the archive.
     * @param resource The classpath resource.
     * @throws java.io.IOException Failed to read resource from classpath.
     */
    public void addEntry(String path, String resource) throws IOException {
        AssertArgument.isNotNull(path, "path");
        AssertArgument.isNotNull(resource, "resource");

        InputStream resourceStream = getClass().getResourceAsStream(resource);
        if(resourceStream == null) {
            throw new IOException("Classpath resource '" + resource + "' no found.");
        } else {
            addEntry(path, resourceStream);
        }
    }

    /**
     * Add the supplied class as an entry in the deployment.
     * @param clazz The class to be added.
     * @throws java.io.IOException Failed to read class from classpath.
     */
    public void addEntry(Class clazz) throws IOException {
        AssertArgument.isNotNull(clazz, "clazz");
        String className = clazz.getName();

        className = className.replace('.', '/') + ".class";
        addEntry(className, "/" + className);
    }

    /**
     * Create an archive of the specified name and containing entries
     * for the data contained in the streams supplied entries arg.
     * specifying the entry name and the value is a InputStream containing
     * the entry data.
     * @param outputStream The archive output stream.
     * @throws java.io.IOException Write failure.
     */
    public void toOutputStream(OutputStream outputStream) throws IOException {
        AssertArgument.isNotNull(outputStream, "outputStream");
        ZipOutputStream archiveStream;

        if(outputStream instanceof ZipOutputStream) {
            archiveStream = (ZipOutputStream) outputStream;
        } else {
            archiveStream = new ZipOutputStream(outputStream);
        }

        try {
            writeEntriesToArchive(archiveStream);
        } finally {
            try {
                archiveStream.flush();
            } finally {
                try {
                    archiveStream.close();
                } catch (IOException e) {
                    logger.info("Unable to close archive output stream.");
                }
            }
        }
    }

    /**
     * Create a {@link java.util.jar.JarInputStream} for the entries defined in this
     * archive.
     * @return The {@link java.util.jar.JarInputStream} for the entries in this archive.
     * @throws java.io.IOException Failed to create stream.
     */
    public JarInputStream toInputStream() throws IOException {
        ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

        toOutputStream(outputStream);

        return new JarInputStream(new ByteArrayInputStream(outputStream.toByteArray()));
    }

    private void writeEntriesToArchive(ZipOutputStream archiveStream) throws IOException {
        Set<Map.Entry<String, byte[]>> entrySet = entries.entrySet();
        for (Map.Entry<String, byte[]> entry : entrySet) {
            try {
                archiveStream.putNextEntry(new ZipEntry(entry.getKey()));
                if(entry.getValue() != null) {
                    archiveStream.write(entry.getValue());
                }
                archiveStream.closeEntry();
            } catch (Exception e) {
                throw (IOException) new IOException("Unable to create archive entry '" + entry.getKey() + "'.").initCause(e);
            }
        }
    }

    private String trimLeadingSlash(String path) {
        StringBuilder builder = new StringBuilder(path);
        while(builder.length() > 0) {
            if(builder.charAt(0) == '/') {
                builder.deleteCharAt(0);
            } else {
                break;
            }
        }
        return builder.toString();
    }
}