/*
 * JBoss, Home of Professional Open Source
 * Copyright 2010, Red Hat Middleware LLC, and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 */
package org.jboss.soa.esb.listeners.message;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.util.Stack;
import java.util.concurrent.BrokenBarrierException;
import java.util.concurrent.CyclicBarrier;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import junit.framework.JUnit4TestAdapter;

import org.jboss.internal.soa.esb.couriers.PickUpOnlyCourier;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.couriers.CourierTimeoutException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
/**
 * Tests for Call modifications while invoking through ServiceInvoker.
 *
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
public class MessageAwareListenerUnitTest
{
    @Before
    public void setUp()
        throws Exception
    {
        MockRegistry.install() ;
    }
    
    @After
    public void tearDown()
        throws Exception
    {
        MockRegistry.uninstall() ;
    }
    
    @Test
    public void testShutdownExecutor()
        throws Exception
    {
        final ConfigTree config = new ConfigTree("MessageAwareListener") ;
        final String category = "TestCategory" ;
        final String service = "TestName" ;
        
        config.setAttribute(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG, category) ;
        config.setAttribute(ListenerTagNames.SERVICE_NAME_TAG, service) ;
        config.setAttribute(ListenerTagNames.MAX_THREADS_TAG, "1") ;
        
        final ConfigTree configEPR = new ConfigTree(ListenerTagNames.EPR_TAG, config) ;
        configEPR.setAttribute(ListenerTagNames.URL_TAG, "invm://test-epr") ;
        
        final ConfigTree actionTree = new ConfigTree(ListenerTagNames.ACTION_ELEMENT_TAG, config) ;
        actionTree.setAttribute(ListenerTagNames.ACTION_ELEMENT_TAG, "TestAction");
        actionTree.setAttribute(ListenerTagNames.ACTION_CLASS_TAG, MessageAwareListenerTestAction.class.getName());

        final MessageAwareListenerMockCourier courier = new MessageAwareListenerMockCourier() ;
        final MessageAwareListenerTestListener listener = new MessageAwareListenerTestListener(config, courier) ;
        courier.setListener(listener) ;
        
        listener.initialise() ;
        listener.start() ;
        
        final boolean result = courier.waitUntilTriggered(10000) ;
        assertTrue("Triggered", result) ;
        
        listener.destroy() ;
        assertNotNull("Message threads", courier.getThread()) ;
        assertNotNull("Execution thread", MessageAwareListenerTestAction.getThread()) ;
        assertEquals("Message and execution threads", courier.getThread(), MessageAwareListenerTestAction.getThread()) ;
    }
    
    @Test
    public void testMaximumThreadPoolCount()
        throws Exception
    {
        final ConfigTree config = new ConfigTree("MessageAwareListener") ;
        final String category = "TestCategory" ;
        final String service = "TestName" ;
        
        config.setAttribute(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG, category) ;
        config.setAttribute(ListenerTagNames.SERVICE_NAME_TAG, service) ;
        config.setAttribute(ListenerTagNames.MAX_THREADS_TAG, "1") ;
        
        final ConfigTree configEPR = new ConfigTree(ListenerTagNames.EPR_TAG, config) ;
        configEPR.setAttribute(ListenerTagNames.URL_TAG, "invm://test-epr") ;
        
        final ConfigTree actionTree = new ConfigTree(ListenerTagNames.ACTION_ELEMENT_TAG, config) ;
        actionTree.setAttribute(ListenerTagNames.ACTION_ELEMENT_TAG, "TestAction");
        actionTree.setAttribute(ListenerTagNames.ACTION_CLASS_TAG, MessageAwareListenerTestThreadCountAction.class.getName());

        final MessageAwareListenerMockThreadCountCourier courier = new MessageAwareListenerMockThreadCountCourier() ;
        final MessageAwareListenerTestListener listener = new MessageAwareListenerTestListener(config, courier) ;
        
        listener.initialise() ;
        
        courier.add(MessageFactory.getInstance().getMessage()) ;
        courier.add(MessageFactory.getInstance().getMessage()) ;
        
        listener.start() ;
        
        sleep(5000) ;
        
        assertEquals("barrier waiting", 1, MessageAwareListenerTestThreadCountAction.BARRIER.getNumberWaiting()) ;
        assertEquals("message count", 1, courier.getMessageCount()) ;
        
        courier.clearMessages() ;
        MessageAwareListenerTestThreadCountAction.BARRIER.reset() ;
        
        listener.setMaximumThreadCount(2) ;

        courier.add(MessageFactory.getInstance().getMessage()) ;
        courier.add(MessageFactory.getInstance().getMessage()) ;
        
        sleep(5000) ;
        
        assertEquals("message count", 0, courier.getMessageCount()) ;
        assertEquals("barrier waiting", 0, MessageAwareListenerTestThreadCountAction.BARRIER.getNumberWaiting()) ;

        listener.setMaximumThreadCount(1) ;

        courier.add(MessageFactory.getInstance().getMessage()) ;
        courier.add(MessageFactory.getInstance().getMessage()) ;
        
        sleep(5000) ;
        
        assertEquals("message count", 1, courier.getMessageCount()) ;
        assertEquals("barrier waiting", 1, MessageAwareListenerTestThreadCountAction.BARRIER.getNumberWaiting()) ;
        
        courier.clearMessages() ;
        MessageAwareListenerTestThreadCountAction.BARRIER.reset() ;
        
        listener.stop() ;
        
        listener.destroy() ;
    }
    
    private void sleep(final long millis)
    {
        try
        {
            Thread.sleep(millis) ;
        }
        catch (final InterruptedException ie) {}
    }
    
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(MessageAwareListenerUnitTest.class) ;
    }
    
    public static final class MessageAwareListenerTestListener extends MessageAwareListener
    {
        private final PickUpOnlyCourier courier ;
        
        public MessageAwareListenerTestListener(final ConfigTree config, final PickUpOnlyCourier courier)
            throws ConfigurationException
        {
            super(config) ;
            this.courier = courier ;
        }

        @Override
        protected PickUpOnlyCourier getCourier()
            throws MalformedEPRException, CourierException
        {
            return courier ;
        }
    }
    
    public static final class MessageAwareListenerTestAction extends AbstractActionPipelineProcessor
    {
        private static Thread thread ;
        
        public MessageAwareListenerTestAction(final ConfigTree config)
        {
        }
        
        public Message process(final Message message)
            throws ActionProcessingException
        {
            setThread() ;
            return message ;
        }
        
        private static synchronized void setThread()
        {
            thread = Thread.currentThread() ;
        }
        
        public static synchronized Thread getThread()
        {
            return thread ;
        }
    }
    
    public static final class MessageAwareListenerMockCourier implements PickUpOnlyCourier
    {
        private MessageAwareListenerTestListener listener ;
        private boolean triggered ;
        private Thread thread ;
        
        public Message pickup(final long millis)
            throws CourierException, CourierTimeoutException
        {
            if (listener != null)
            {
                setThread() ;
                try
                {
                    listener.stop() ;
                }
                catch (final ManagedLifecycleException mle)
                {
                    throw new CourierException("Unexpected lifecycle exception", mle) ;
                }
                synchronized(this)
                {
                    triggered = true ;
                    notify() ;
                }
            }
            return MessageFactory.getInstance().getMessage() ;
        }

        public boolean waitUntilTriggered(final long millis)
            throws InterruptedException
        {
            final long end = System.currentTimeMillis() + millis ;
            do
            {
                final long delay = end - System.currentTimeMillis() ;
                if (delay > 0)
                {
                    synchronized(this)
                    {
                        if (!triggered)
                        {
                            wait(delay) ;
                        }
                        if (triggered)
                        {
                            return true ;
                        }
                    }
                }
                else
                {
                    return false ;
                }
            }
            while(true) ;
        }

        public void cleanup()
        {
        }

        public void setListener(final MessageAwareListenerTestListener listener)
        {
            this.listener = listener ;
        }
        
        private synchronized void setThread()
        {
            thread = Thread.currentThread() ;
        }
        
        public synchronized Thread getThread()
        {
            return thread ;
        }
    }

    public static final class MessageAwareListenerMockThreadCountCourier implements PickUpOnlyCourier
    {
        private final Stack<Message> messages = new Stack<Message>() ;
        
        public Message pickup(final long millis)
            throws CourierException, CourierTimeoutException
        {
            if (messages.empty())
            {
                return null ;
            }
            else
            {
                return messages.pop() ;
            }
        }
        
        public void add(final Message message)
        {
            messages.push(message) ;
        }
        
        public int getMessageCount()
        {
            return messages.size() ;
        }
        
        public void clearMessages()
        {
            messages.clear() ;
        }

        public void cleanup()
        {
        }
    }
    
    public static final class MessageAwareListenerTestThreadCountAction extends AbstractActionPipelineProcessor
    {
        public static CyclicBarrier BARRIER = new CyclicBarrier(2) ;
        
        public MessageAwareListenerTestThreadCountAction(final ConfigTree config)
        {
        }
        
        public Message process(final Message message)
            throws ActionProcessingException
        {
            try
            {
                    BARRIER.await(20, TimeUnit.SECONDS) ;
            }
            catch (final TimeoutException te) {} // ignore
            catch (final InterruptedException ie) {} // ignore
            catch (final BrokenBarrierException bbe) {} // ignore
            
            return message ;
        }
    }
}
