package org.jboss.soa.esb.util;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;

import junit.framework.TestCase;

import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.notification.NotificationException;
import org.jboss.soa.esb.notification.NotificationTarget;

/**
 * Mock NotificationTarget Implementation. <p/> Configured by giving the target
 * output list a 'name'. Notifications are statically accessed via the static
 * {@link #getTargetList(String)} method, supplying the target list name. <p/>
 * Sample config:
 * 
 * <pre>
 *  &lt;NotificationList type=&quot;OK&quot;&gt;
 *  	&lt;target class=&quot;org.jboss.soa.esb.util.MockNotificationTarget&quot; &lt;b&gt;name=&quot;ok-target&quot;&lt;/b&gt; /&gt;
 *  &lt;/NotificationList&gt;
 * </pre>
 * 
 * @author tfennelly
 */
public class MockNotificationTarget extends NotificationTarget
{

	private static Hashtable<String, List<Serializable>> targetLists = new Hashtable<String, List<Serializable>>();

	private List<Serializable> targetList;

	public MockNotificationTarget (ConfigTree config)
	{
		super(config);

		String name = config.getAttribute("name");

		if (name == null || name.trim().equals(""))
		{
			TestCase
					.fail("Mock NotificationTarget configured incorrectly.  Must specify a 'name' attribute on the NotificationList/target element.");
		}

		targetList = getTargetList(name);
	}

	public static List<Serializable> getTargetList (String name)
	{
		synchronized (targetLists)
		{
			List<Serializable> notificationList = targetLists.get(name);

			// Never return a null list.
			if (notificationList == null)
			{
				notificationList = new ArrayList<Serializable>();
				targetLists.put(name, notificationList);
			}

			return notificationList;
		}
	}

	public static void clearNotifications ()
	{
		synchronized (targetLists)
		{
			targetLists.clear();
		}
	}

	@Override
	public void sendNotification (Message notificationObject)
			throws NotificationException
	{
		targetList.add(notificationObject.getBody().get().toString());
	}
}
