/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.workitem.esb;

import java.net.URI;
import java.util.Map;

import org.apache.log4j.Logger;

import org.drools.runtime.process.WorkItem;
import org.drools.runtime.process.WorkItemHandler;
import org.drools.runtime.process.WorkItemManager;

import org.jboss.internal.soa.esb.addressing.helpers.EPRHelper;
import org.jboss.internal.soa.esb.util.LRUCache;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.PortReference;
import org.jboss.soa.esb.addressing.eprs.LogicalEPR;
import org.jboss.soa.esb.addressing.MalformedEPRException;
import org.jboss.soa.esb.client.ServiceInvoker;
import org.jboss.soa.esb.couriers.Courier;
import org.jboss.soa.esb.couriers.CourierException;	
import org.jboss.soa.esb.couriers.CourierFactory;
import org.jboss.soa.esb.couriers.CourierUtil;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;

import org.jboss.soa.esb.services.jbpm5.model.ProcessConstants;

public class ESBActionWorkItemHandler implements WorkItemHandler {

    public static final String REPLY_TO                     = "jbpmReplyTo";
    public static final String FAULT_TO                     = "jbpmFaultTo";
    public static final String ESB_MESSAGE_ID               = "jbpmEsbMessageId";

	public static final String PROCESS_ID                   = "jbpmProcessId";
	public static final String PROCESS_SCOPE_ATTR 			= "process-scope";

    public String replyToOriginator;
    public String esbCategoryName;
    public String esbServiceName;
    public String jbpmSessionId;
    public String callbackCategoryName;
    public String callbackServiceName;
    
    public Boolean globalProcessScope;

    public String exceptionTransition;

    public Integer millisToWaitForResponse;

    private transient Logger logger = Logger.getLogger(ESBActionWorkItemHandler.class);
    private static transient LRUCache<String, ServiceInvoker> siCache = new LRUCache<String, ServiceInvoker>(20);
    
    /**
     * Retrieve category/service/session ID information from the workItem - as well as the 
     * callback category and service information - and 
     * @param workItem
     * @param manager
     */
    public void executeWorkItem(WorkItem workItem, WorkItemManager manager) {

    	esbCategoryName = (String) workItem.getParameter("ServiceCategory");
    	esbServiceName = (String) workItem.getParameter("ServiceName");
    	jbpmSessionId = (String) workItem.getParameter("jbpm5-session-id");
        callbackCategoryName = (String) workItem.getParameter("CallbackServiceCategory");
        callbackServiceName = (String) workItem.getParameter("CallbackServiceName");
        if (! "".equals(workItem.getParameter("replyToOriginator"))) {
            replyToOriginator = (String) workItem.getParameter("replyToOriginator");
        }
        
        if (replyToOriginator != null) {
            if (!(ProcessConstants.EPR_REPLY.equals(replyToOriginator) 
            		|| ProcessConstants.EPR_FAULT.equals(replyToOriginator))) {            	
            	logger.error("EPR type (replyToOriginator) must be \"" 
            			+ ProcessConstants.EPR_REPLY + "\" or \"" 
            			+ ProcessConstants.EPR_FAULT + "\"");
            }
        } else {
            if (null == esbCategoryName)
            	logger.error("Service category (ServiceCategory element) must not be null");
            if (null == esbServiceName)
            	logger.error("Service name (ServiceName element) must not be null");
        }
    	
        Message message = MessageFactory.getInstance().getMessage();
        Body body = message.getBody();
        body.add("");

        Map<String,Object> parameters = (Map<String,Object>)workItem.getParameters();
        if (parameters != null) {
            for (Map.Entry<String,Object> entry : parameters.entrySet()) {
                body.add((String)entry.getKey(), entry.getValue());
            }
        }

        EPR replyTo = createReplyTo(workItem, manager);
        message.getHeader().getCall().setReplyTo(replyTo);
        if (exceptionTransition!=null) {
        	// Set the replyTo to the JBpmCallback Service
            EPR faultTo = createFaultTo(workItem, manager);
            message.getHeader().getCall().setFaultTo(faultTo);
        }
        
        if (isReplyToOrFaultToSet(workItem))
        {
            setRelatesToMessageId(workItem, message);
        }
        
        //Sending the message on its way
        if (logger.isDebugEnabled()) logger.debug("Created ESB message=" + message);
        
        if (replyToOriginator != null) {
            EPR epr = null;
            final Object replyToEPR = workItem.getParameters().get(REPLY_TO);
            final Object faultToEPR = workItem.getParameters().get(FAULT_TO);

            if (ProcessConstants.EPR_FAULT.equals(replyToOriginator) && (faultToEPR != null)) {
            	try {
            		epr = EPRHelper.fromXMLString(faultToEPR.toString()) ;
            	} catch (org.jboss.soa.esb.UnmarshalException ue) {
            		logger.error("Could not create FaultToEPR");
            	}
            } else if (replyToEPR != null) {
            	try {
            		epr = EPRHelper.fromXMLString(replyToEPR.toString()) ;
            	} catch (org.jboss.soa.esb.UnmarshalException ue) {
            		logger.error("Could not create ReplyToEPR");
            	}
            } 
                        
            if(epr instanceof LogicalEPR) {
            	try {
            		final ServiceInvoker invoker = ((LogicalEPR)epr).getServiceInvoker();
                    invoker.deliverAsync(message);
            	} catch (MessageDeliverException mde) {
            		logger.error("Could not retrieve/deliverAsync LogicalEPR with ServiceInvoker", mde);
            	}
            } else {
            	Courier courier = null;
                try {
                    courier = CourierFactory.getCourier(epr);
                	courier.deliver(message);
                } catch (CourierException ce) {
                	logger.error("Could not deliver message with courier", ce);
                } catch (MalformedEPRException mee) {
                	logger.error("MalformedEPR " + epr, mee);
                } finally {
                	if (courier != null) {
                		CourierUtil.cleanCourier(courier);
                	}
                }
            }
        } else {
        	try {
        		getServiceInvoker().deliverAsync(message);
        	} catch (MessageDeliverException mde) {
        		logger.error("Could not deliver message", mde);
            	manager.abortWorkItem(workItem.getId());
        	}
        }
        logger.debug("Message send successfully");        
    }

    /**
     * Empty abortWorkItem implementation.
     * @param workItem work item
     * @param manager manager
     */
    public void abortWorkItem(WorkItem workItem, WorkItemManager manager) {
    }
    
    /**
     * Caches the most recently used ServiceInvokers.
     * 
     * @return a ServiceInvoker for the current esbService and esbCategoryName.
     * @throws MessageDeliverException
     */
    private ServiceInvoker getServiceInvoker() throws MessageDeliverException
    {
        String key = esbCategoryName + esbServiceName;
        final ServiceInvoker origInvoker = siCache.get(key) ;
        if (origInvoker != null) {
            return origInvoker;
        } else {
            System.setProperty("javax.xml.registry.ConnectionFactoryClass","org.apache.ws.scout.registry.ConnectionFactoryImpl");

            ServiceInvoker invoker = new ServiceInvoker(esbCategoryName,  esbServiceName);
            siCache.put(key, invoker);
            return invoker;
        }
    }
    
    /**
     * Setup the replyTo for the CallBack Service.
     * 
     * Creates or increments a process-node-version-counter whose name is related to 
     * the current node ID. The name of the counter is jbpmProcessNodeVersionCounter<NodeId>.
     * The counter is added to the ProcessVariable Map (global to the ProcessInstance) 
     * on the jBPM side The same variable should be added to the 
     * EsbMessage before it is passed onto the ESB. Other parameters added
     * are the returnVariable mapping, the nodeId and the tokenId.
     * 
     * @param workItem the work item
     * @param manager the work item manager
     * @return the replyTo EPR of the JBpmCallbackService.
     */
    protected EPR createReplyTo(WorkItem workItem, WorkItemManager manager)
    {
        EPR replyTo = new LogicalEPR(callbackCategoryName, callbackServiceName);
        PortReference portReference = replyTo.getAddr();
        if (globalProcessScope != null) {
            portReference.addExtension(PROCESS_SCOPE_ATTR, globalProcessScope.toString());
        }
        
        String workItemId = String.valueOf(workItem.getId());
        portReference.addExtension(ProcessConstants.WORK_ITEM_ID, workItemId);
        portReference.addExtension(ProcessConstants.SESSION_ID, jbpmSessionId);
        portReference.addExtension(ProcessConstants.PROCESS_INSTANCE_ID, String.valueOf(workItem.getProcessInstanceId()));
        return replyTo;
    }
    
    /**
     * Sets the faultTo EPR. This way jBPM can handle a failure in the ESB service by taking a 
     * exception (faultTo) transition.
     * @param workItem the work item
     * @param manager the work item manager
     * @return the faultTo EPR of the JBpmCallbackService.
     */
    protected EPR createFaultTo(WorkItem workItem, WorkItemManager manager)
    {
        EPR faultTo = createReplyTo(workItem, manager);
        if (!exceptionTransition.equals("condition")) {
            String workItemId = String.valueOf(workItem.getId());

        	PortReference portReference = faultTo.getAddr();
            portReference.addExtension(ProcessConstants.WORK_ITEM_ID, workItemId);
            portReference.addExtension(ProcessConstants.SESSION_ID, jbpmSessionId);
            portReference.addExtension(ProcessConstants.PROCESS_INSTANCE_ID, String.valueOf(workItem.getProcessInstanceId()));
        }
        return faultTo;
    }
    
    /**
     * Returns true/false whether the REPLY_TO or FAULT_TO parameters are set on the workItem.
     * @param workItem work item
     * @return whether the either reply_to or fault_to are set
     */
    public static boolean isReplyToOrFaultToSet(final WorkItem workItem)
    {
        return workItem.getParameter(REPLY_TO) != null || workItem.getParameter(FAULT_TO) != null;
    }
    
    /**
     * Sets the relates to message ID in the header.
     * @param workItem work item
     * @param message message
     */
    public static void setRelatesToMessageId(final WorkItem workItem, final Message message)
    {
        final URI esbMessageId = (URI) workItem.getParameter(ESB_MESSAGE_ID);
        message.getHeader().getCall().setRelatesTo(esbMessageId);
    }
}
