/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.rules;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.Properties;

import junit.framework.JUnit4TestAdapter;

import org.drools.KnowledgeBase;
import org.drools.SystemEventListener;
import org.drools.SystemEventListenerFactory;
import org.drools.agent.KnowledgeAgent;
import org.drools.agent.RuleAgent;
import org.drools.compiler.DroolsParserException;
import org.drools.definition.KnowledgePackage;
import org.jboss.soa.esb.services.rules.RuleInfo;
import org.junit.AfterClass;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Unit test for {@link DroolsRuleBaseHelper}
 * <p/>
 * 
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class DroolsRuleBaseHelperUnitTest
{
	// instance under test
	private final static DroolsRuleBaseHelper helper = DroolsRuleBaseHelper.getInstance();
	
	private final static String NULL_DSL_FILE = null;
	private final static String NULL_RULE_FILE = null;
	private final static String NULL_DECISION_TABLE_FILE = null;
	
	private static SystemEventListener originalSystemEventListener = null;
	
	@BeforeClass
	public static void beforeClass()
	{
		originalSystemEventListener = SystemEventListenerFactory.getSystemEventListener();
		SystemEventListenerFactory.setSystemEventListener(new LogSystemEventListener());
	}
	
	@AfterClass
	public static void afterClass()
	{
		SystemEventListenerFactory.setSystemEventListener(originalSystemEventListener);
	}
	
	@Test ( expected = IllegalArgumentException.class )
	public void shouldThrowIfRuleFileIsNull() throws RuleServiceException
	{
		RuleInfo ruleInfo = new RuleInfoBuilder(NULL_RULE_FILE).dslSource(NULL_DSL_FILE).build();
		helper.createRuleBaseFromRuleFiles( ruleInfo );
	}
	
	@Test
	public void createRuleBaseFromRuleFiles() throws RuleServiceException
	{
		RuleInfo ruleInfo = new RuleInfoBuilder("RuleBaseHelper.drl").dslSource(NULL_DSL_FILE).build();
		KnowledgeBase ruleBase = helper.createRuleBaseFromRuleFiles( ruleInfo );
		
		assertNotNull( ruleBase );
		assertTrue( ruleBase.getKnowledgePackages().size() > 0 );
		assertEquals( "org.jboss.internal.soa.esb.rules" , ruleBase.getKnowledgePackages().iterator().next().getName());
	}
	
	@Test ( expected = IllegalArgumentException.class )
	public void shouldThrowIfDecisionTableIsNull() throws DroolsParserException, IOException, RuleServiceException
	{
		RuleInfo ruleInfo = new RuleInfoBuilder(NULL_DECISION_TABLE_FILE).build();
		helper.createRuleBaseFromDecisionTable( ruleInfo );
	}
	
	@Test
	public void createRuleBaseFromDecisionTable() throws RuleServiceException
	{
		RuleInfo ruleInfo = new RuleInfoBuilder("RuleBaseHelper.xls").build();
		KnowledgeBase ruleBase = helper.createRuleBaseFromDecisionTable( ruleInfo );
		assertNotNull( ruleBase );
		assertEquals( "org.jboss.internal.soa.esb.rules", ruleBase.getKnowledgePackages().iterator().next().getName());
	}
	
	@Test ( expected = NullPointerException.class )
	public void getRuleAsStringShouldThrowIfRuleFileIsNull() throws RuleServiceException
	{
		String rulesAsString = helper.getRulesAsString( NULL_RULE_FILE, NULL_DSL_FILE );
		assertNotNull( rulesAsString );
	}
	
	@Test
	public void getRuleFileAsString() throws RuleServiceException
	{
		String rulesAsString = helper.getRulesAsString( "RuleBaseHelper.drl", NULL_DSL_FILE );
		assertNotNull( rulesAsString );
	}
	
	@Test
	public void getSpreadSheetRules() throws RuleServiceException
	{
		RuleInfo ruleInfo = new RuleInfoBuilder("RuleBaseHelper.xls").build();
		KnowledgeBase spreadSheetRuleBase = helper.createRuleBaseFromDecisionTable( ruleInfo );
		assertNotNull( spreadSheetRuleBase );
	}
	
	@Test
	public void verifyRuleAgentPkgFile() throws Exception
	{
		KnowledgePackage kpkg;
		File file = null;
		OutputStream os = null;
		try
		{
			Class<?> clazz = getClass();
			file = File.createTempFile(clazz.getSimpleName() + "-", ".properties");
			Properties properties = new Properties();
			properties.setProperty(RuleAgent.URLS, clazz.getResource("/testrules.pkg").toString());
			os = new FileOutputStream(file);
			properties.store(os, clazz.getSimpleName());
			RuleInfo ruleInfo = new RuleInfoBuilder(file.getAbsolutePath()).build();
			KnowledgeAgent ka = helper.createRuleAgent(ruleInfo);
		    kpkg = ka.getKnowledgeBase().getKnowledgePackage("org.jboss.internal.soa.esb.rules");

		}
		finally
		{
			try { if (os != null) os.close(); } catch (Throwable t) {}
			try { if (file != null) file.delete(); } catch (Throwable t) {}
		}
	    assertNotNull(kpkg);
	    assertEquals("org.jboss.internal.soa.esb.rules", kpkg.getName());
	}
	
	public static junit.framework.Test suite()
	{
		return new JUnit4TestAdapter( DroolsRuleBaseHelperUnitTest.class );
	}
	
}
