/*
 * RHQ Management Platform
 * Copyright (C) 2005-2008 Red Hat, Inc.
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
package org.jbosson.plugins.jbossesb;

import java.util.Set;

import org.mc4j.ems.connection.bean.attribute.EmsAttribute;
import org.rhq.core.domain.measurement.MeasurementDataNumeric;
import org.rhq.core.domain.measurement.MeasurementReport;
import org.rhq.core.domain.measurement.MeasurementScheduleRequest;
import org.rhq.core.pluginapi.inventory.ResourceContext;
import org.rhq.plugins.jmx.MBeanResourceComponent;

import org.rhq.core.domain.configuration.Configuration;
import org.rhq.core.domain.configuration.PropertySimple;

import org.rhq.core.pluginapi.event.EventContext;


/**
 * The ServiceComponent finds ESB service data (messages, bytes, time).
 * 
 * @author Tom Cunningham
 */
public class ServiceComponent extends MBeanResourceComponent<MBeanResourceComponent> {
    private ResourceContext<MBeanResourceComponent> context;
    
	private static final String OVERALL_TIME_PROCESSED = "overall service time processed";
	private static final String OVERALL_TIME_PROCESSED_METRIC = "overallTimeProcessed";

    private static final String OVERALL_MESSAGE_COUNT = "overall service message count";    
    private static final String OVERALL_MESSAGE_METRIC = "overallMessageCount";
   
    private static final String OVERALL_BYTES_METRIC = "overallBytes";
    private static final String OVERALL_BYTES_COUNT = "overall service count bytes";
    
    private static final String OVERALL_BYTES_PROCESSED_METRIC = "bytesProcessed";
    private static final String OVERALL_BYTES_FAILED_METRIC = "bytesFailed";
    
    private static final String OVERALL_BYTES_PROCESSED = "overall processedbytes";
    private static final String OVERALL_BYTES_FAILED = "overall failed bytes";

    private static final String OVERALL_MINUTE_METRIC_NAME = "overallMessageCountByMinute";
    
    EventContext eventContext;
    public static final String ESB_MESSAGE_EVENT = "ESBMessageAlert";

    
    @Override
    public void start(ResourceContext<MBeanResourceComponent> context) {
        super.start(context);
        this.context = context;
        eventContext = context.getEventContext();

        ESBMessageEventPoller ep = new ESBMessageEventPoller(ESB_MESSAGE_EVENT);
        if (ep.setConnection(getEmsConnection())) {
            eventContext.registerEventPoller(ep, 53);
        }
    }

    @Override
    public void getValues(MeasurementReport report, Set<MeasurementScheduleRequest> requests) {
    	Configuration pluginConfig = this.context.getPluginConfiguration();
    	pluginConfig.put(new PropertySimple("type", "service"));
    	Integer overallMessages = new Integer(0);
    	
    	long bytesProcessed = 0;
    	long bytesFailed = 0;
        for (MeasurementScheduleRequest request : requests) {
        	String metricName = request.getName();
        	if (metricName.equals(OVERALL_MESSAGE_METRIC)) {
        		EmsAttribute attribute = getEmsBean().getAttribute(OVERALL_MESSAGE_COUNT);
        		overallMessages = (Integer) attribute.refresh();
        		report.addData(new MeasurementDataNumeric(request, new Double(overallMessages.doubleValue())));
        	} else if (metricName.equals(OVERALL_TIME_PROCESSED_METRIC)) {
        		EmsAttribute attribute = getEmsBean().getAttribute(OVERALL_TIME_PROCESSED);
        		Long overallTimeProcessed = (Long) attribute.refresh();
        		if (overallTimeProcessed != null) {
        			Double timeProcessed = new Double(overallTimeProcessed.doubleValue() / 1000000);
        			report.addData(new MeasurementDataNumeric(request, timeProcessed));
        		} else {
        			report.addData(new MeasurementDataNumeric(request, new Double(0)));
        		}
        	} else if (metricName.equals(OVERALL_BYTES_PROCESSED_METRIC)) {
        		EmsAttribute attribute = getEmsBean().getAttribute(OVERALL_BYTES_PROCESSED);
        		Long processed = new Long(0);
        		if (attribute != null) {
        			processed = (Long) attribute.refresh();
        			bytesProcessed = processed.longValue();
        		}
        		report.addData(new MeasurementDataNumeric(request, new Double(processed.doubleValue())));
        	} else if (metricName.equals(OVERALL_BYTES_FAILED_METRIC)) {
        		EmsAttribute attribute = getEmsBean().getAttribute(OVERALL_BYTES_FAILED);
        		Long processed = new Long(0);
        		if (attribute != null) {
        			processed = (Long) attribute.refresh();
        			bytesFailed = processed.longValue();
        		}
        		report.addData(new MeasurementDataNumeric(request, new Double(processed.doubleValue())));        		
        	} else if (metricName.equals(OVERALL_BYTES_METRIC)) {
        		EmsAttribute attribute = getEmsBean().getAttribute(OVERALL_BYTES_COUNT);
        		Long processed = new Long(0);
        		if (attribute != null) {
        			processed = (Long) attribute.refresh();
        		}
        		report.addData(new MeasurementDataNumeric(request, new Double(processed.doubleValue())));
        	}
        }

        for (MeasurementScheduleRequest request : requests) {
        	String metricName = request.getName();
        	if (metricName.equals(OVERALL_MINUTE_METRIC_NAME)) {
        		report.addData(new MeasurementDataNumeric(request, new Double(overallMessages.doubleValue())));
        	}
        }
    }
}
