/*
 * Copyright 2022 Red Hat, Inc. and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.kie.efesto.common.api.io;

import org.kie.efesto.common.api.exceptions.KieEfestoCommonException;
import org.kie.efesto.common.api.utils.FileNameUtils;

import java.io.File;

import static org.kie.efesto.common.api.utils.FileNameUtils.getFileName;


/**
 * This specific <code>File</code> is meant to contain the index of the classes generated by a <b>compilation-plugin</b>
 */
public final class IndexFile extends File {

    public static final String INDEX_FILE = "IndexFile";
    public static final String FINAL_SUFFIX = "_json";

    static String getIndexFileName(String modelType) {
        return String.format("%s.%s%s", INDEX_FILE, modelType, FINAL_SUFFIX);
    }

    static String validatePathName(String toValidate) {
        String fileName = getFileName(toValidate);
        if (!fileName.endsWith(FINAL_SUFFIX)) {
            throw new KieEfestoCommonException("Wrong file name " + fileName);
        }
        String model = getModel(fileName);
        if (model.isEmpty()) {
            throw new KieEfestoCommonException("Wrong file name " + fileName);
        }
        return toValidate;
    }

    static String getModel(String fileName) {
        return FileNameUtils.getSuffix(fileName).replace(FINAL_SUFFIX, "");
    }

    public IndexFile(String modelType) {
        super(validatePathName(getIndexFileName(modelType)));
    }

    public IndexFile(String parent, String modelType) {
        super(parent, validatePathName(getIndexFileName(modelType)));
    }

    public IndexFile(File existingFile) {
        super(existingFile.toURI());
    }

    public String getModel() {
        return getModel(getSuffix());
    }

    private String getSuffix() {
        return FileNameUtils.getSuffix(this.getName());
    }

}
