/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.drools.core.reteoo;

import org.drools.base.base.ClassObjectType;
import org.drools.base.base.ObjectType;
import org.drools.base.common.RuleBasePartitionId;
import org.drools.base.reteoo.NodeTypeEnums;
import org.drools.base.rule.Pattern;
import org.drools.core.common.BaseNode;
import org.drools.core.phreak.BuildtimeSegmentUtilities;
import org.drools.core.reteoo.builder.BuildContext;
import org.drools.util.bitmask.AllSetBitMask;
import org.drools.util.bitmask.BitMask;
import org.drools.util.bitmask.EmptyBitMask;

import java.util.Collection;
import java.util.List;

import static org.drools.base.reteoo.PropertySpecificUtil.calculateNegativeMask;
import static org.drools.base.reteoo.PropertySpecificUtil.calculatePositiveMask;
import static org.drools.base.reteoo.PropertySpecificUtil.getAccessibleProperties;
import static org.drools.base.reteoo.PropertySpecificUtil.isPropertyReactive;

/**
 * A source of <code>ReteTuple</code> s for a <code>TupleSink</code>.
 *
 * <p>
 * Nodes that propagate <code>Tuples</code> extend this class.
 * </p>
 *
 * @see LeftTupleSource
 * @see LeftTuple
 */
public abstract class LeftTupleSource extends BaseNode implements LeftTupleNode {

    protected BitMask                 leftDeclaredMask = EmptyBitMask.get();
    protected BitMask                 leftInferredMask = EmptyBitMask.get();
    protected BitMask                 leftNegativeMask = EmptyBitMask.get();


    /** The left input <code>TupleSource</code>. */
    protected LeftTupleSource         leftInput;



    /** The destination for <code>Tuples</code>. */
    protected LeftTupleSinkPropagator sink;

    private transient ObjectTypeNode.Id leftInputOtnId;

    private int pathIndex;

    private int objectCount;

    // ------------------------------------------------------------
    // Constructors
    // ------------------------------------------------------------
    public LeftTupleSource() {

    }

    /**
     * Single parameter constructor that specifies the unique id of the node.
     *
     * @param id
     */
    protected LeftTupleSource(int id, BuildContext context) {
        super(id, context != null ? context.getPartitionId() : RuleBasePartitionId.MAIN_PARTITION);
        this.sink = EmptyLeftTupleSinkAdapter.getInstance();
        initMemoryId( context );
    }

    // ------------------------------------------------------------
    // Instance methods
    // ------------------------------------------------------------

    public int getPathIndex() {
        return pathIndex;
    }

    public abstract short getType();

    public abstract LeftTuple createPeer(LeftTuple original);

    public void addTupleSink(final LeftTupleSink tupleSink) {
        addTupleSink(tupleSink, null);
    }

    public LeftTupleSource getLeftTupleSource() {
        return leftInput;
    }

    public final void setLeftTupleSource(LeftTupleSource leftInput) {
        this.leftInput = leftInput;
        pathIndex = leftInput.getPathIndex() + 1;
    }

    public int getObjectCount() {
        return objectCount;
    }

    public void setObjectCount(int count) {
        objectCount = count;
    }

    /**
     * Adds the <code>TupleSink</code> so that it may receive
     * <code>Tuples</code> propagated from this <code>TupleSource</code>.
     *
     * @param tupleSink
     *            The <code>TupleSink</code> to receive propagated
     *            <code>Tuples</code>.
     */
    public void addTupleSink(final LeftTupleSink tupleSink, final BuildContext context) {
        this.sink = addTupleSink(this.sink, tupleSink, context);
    }

    protected LeftTupleSinkPropagator addTupleSink(final LeftTupleSinkPropagator sinkPropagator, final LeftTupleSink tupleSink, final BuildContext context) {
        if ( sinkPropagator instanceof EmptyLeftTupleSinkAdapter ) {
            // otherwise, we use the lighter synchronous propagator
            return new SingleLeftTupleSinkAdapter( this.getPartitionId(), tupleSink );
        }

        if ( sinkPropagator instanceof SingleLeftTupleSinkAdapter ) {
            CompositeLeftTupleSinkAdapter sinkAdapter = new CompositeLeftTupleSinkAdapter( this.getPartitionId() );
            sinkAdapter.addTupleSink( sinkPropagator.getSinks()[0] );
            sinkAdapter.addTupleSink( tupleSink );
            return sinkAdapter;
        }

        ((CompositeLeftTupleSinkAdapter) sinkPropagator).addTupleSink( tupleSink );
        return sinkPropagator;
    }

    /**
     * Removes the <code>TupleSink</code>
     *
     * @param tupleSink
     *            The <code>TupleSink</code> to remove
     */
    public void removeTupleSink(final LeftTupleSink tupleSink) {
        if ( this.sink instanceof EmptyLeftTupleSinkAdapter ) {
            throw new IllegalArgumentException( "Cannot remove a sink, when the list of sinks is null" );
        }

        if ( this.sink instanceof SingleLeftTupleSinkAdapter ) {
            this.sink = EmptyLeftTupleSinkAdapter.getInstance();
        } else {
            final CompositeLeftTupleSinkAdapter sinkAdapter = (CompositeLeftTupleSinkAdapter) this.sink;
            sinkAdapter.removeTupleSink( tupleSink );
            if ( sinkAdapter.size() == 1 ) {
                this.sink = new SingleLeftTupleSinkAdapter( this.getPartitionId(), sinkAdapter.getSinks()[0] );
            }
        }
    }

    public LeftTupleSinkNode getFirstLeftTupleSinkIgnoreRemoving(TerminalNode removingTn) {
        if (removingTn == null) {
            return this.sink.getFirstLeftTupleSink();
        }

        for ( LeftTupleSink sink : this.sink.getSinks()) {
            if ( !BuildtimeSegmentUtilities.sinkNotExclusivelyAssociatedWithTerminal(removingTn, sink)) {
                // skip this node as it's being removed;
                continue;
            }
            return (LeftTupleSinkNode) sink;
        }

        throw new RuntimeException("This should always return a sink");
    }

    public LeftTupleSinkPropagator getSinkPropagator() {
        return this.sink;
    }

    public void setSourcePartitionId(BuildContext context, RuleBasePartitionId partitionId) {
        setSourcePartitionId(leftInput, context, partitionId);
    }

    protected void setSourcePartitionId(BaseNode source, BuildContext context, RuleBasePartitionId partitionId) {
        if (this.partitionId == partitionId) {
            return;
        }
        this.partitionId = partitionId;
        if (source.getPartitionId() == RuleBasePartitionId.MAIN_PARTITION) {
            setPartitionIdWithSinks( partitionId );
        } else {
            source.setPartitionId( context, partitionId );
        }
    }

    public final void setPartitionIdWithSinks( RuleBasePartitionId partitionId ) {
        this.partitionId = partitionId;
        for (LeftTupleSink sink : getSinkPropagator().getSinks()) {
            sink.setPartitionIdWithSinks( partitionId );
        }
    }

    public boolean isInUse() {
        return this.sink.size() > 0;
    }

    protected final void initMasks(BuildContext context,
                                   LeftTupleSource leftInput) {
        initDeclaredMask( context, leftInput );
        initInferredMask( leftInput );
    }

    protected void initDeclaredMask(BuildContext context,
                                    LeftTupleSource leftInput) {
        if ( context == null || context.getLastBuiltPatterns() == null ) {
            // only happens during unit tests
            leftDeclaredMask = AllSetBitMask.get();
            return;
        }

        if ( leftInput.getType() != NodeTypeEnums.LeftInputAdapterNode) {
            // BetaNode's not after LIANode are not relevant for left mask property specific, so don't block anything.
            leftDeclaredMask = AllSetBitMask.get();
            return;
        }

        Pattern pattern = getLeftInputPattern( context ); // left input pattern

        ObjectType objectType = getObjectTypeForPropertyReactivity( (LeftInputAdapterNode) leftInput, pattern );

        if ( !(objectType instanceof ClassObjectType) ) {
            // Only ClassObjectType can use property specific
            leftDeclaredMask = AllSetBitMask.get();
            return;
        }

        if ( pattern != null && isPropertyReactive(context.getRuleBase(), objectType) ) {
            Collection<String> leftListenedProperties = pattern.getListenedProperties();
            List<String> accessibleProperties = getAccessibleProperties( context.getRuleBase(), objectType );
            leftDeclaredMask = calculatePositiveMask( objectType, leftListenedProperties, accessibleProperties );
            leftDeclaredMask = setNodeConstraintsPropertyReactiveMask(leftDeclaredMask, objectType, accessibleProperties);
            leftNegativeMask = calculateNegativeMask( objectType, leftListenedProperties, accessibleProperties );
            setLeftListenedProperties(leftListenedProperties);
        } else {
            // if property specific is not on, then accept all modification propagations
            leftDeclaredMask = AllSetBitMask.get();
        }
    }

    protected BitMask setNodeConstraintsPropertyReactiveMask(BitMask mask, ObjectType objectType, List<String> accessibleProperties) {
        return mask;
    }

    protected Pattern getLeftInputPattern( BuildContext context ) {
        return context.getLastBuiltPatterns()[1];
    }

    protected ObjectType getObjectTypeForPropertyReactivity( LeftInputAdapterNode leftInput, Pattern pattern ) {
        return pattern != null ?
               pattern.getObjectType() :
               leftInput.getParentObjectSource().getObjectTypeNode().getObjectType();
    }

    protected void setLeftListenedProperties(Collection<String> leftListenedProperties) { }

    protected void initInferredMask(LeftTupleSource leftInput) {
        LeftTupleSource unwrappedLeft = unwrapLeftInput(leftInput);
        if ( unwrappedLeft.getType() == NodeTypeEnums.LeftInputAdapterNode && ((LeftInputAdapterNode)unwrappedLeft).getParentObjectSource().getType() == NodeTypeEnums.AlphaNode ) {
            ObjectSource objectSource = ((LeftInputAdapterNode)unwrappedLeft).getParentObjectSource();
            leftInferredMask = objectSource.updateMask( leftDeclaredMask );
        } else {
            leftInferredMask = leftDeclaredMask;
        }
        leftInferredMask = leftInferredMask.resetAll(leftNegativeMask);
    }

    private LeftTupleSource unwrapLeftInput(LeftTupleSource leftInput) {
        if (leftInput.getType() == NodeTypeEnums.FromNode || leftInput.getType() == NodeTypeEnums.ReactiveFromNode) {
            return leftInput.getLeftTupleSource();
        }
        return leftInput;
    }

    public BitMask getLeftDeclaredMask() {
        return leftDeclaredMask;
    }

    public BitMask getLeftInferredMask() {
        return leftInferredMask;
    }

    protected void setLeftInferredMask(BitMask leftInferredMask) {
        this.leftInferredMask = leftInferredMask;
    }

    public BitMask getLeftNegativeMask() {
        return leftNegativeMask;
    }

    public ObjectTypeNode.Id getLeftInputOtnId() {
        return leftInputOtnId;
    }

    public void setLeftInputOtnId(ObjectTypeNode.Id leftInputOtnId) {
        this.leftInputOtnId = leftInputOtnId;
    }

    public ObjectType getObjectType() {
        ObjectTypeNode objectTypeNode = getObjectTypeNode();
        return objectTypeNode != null ? objectTypeNode.getObjectType() : null;
    }

    public abstract boolean isLeftTupleMemoryEnabled();
}
