#include <glib.h>


#define BD_DM_ERROR bd_dm_error_quark ()
typedef enum {
    BD_DM_ERROR_SYS,
    BD_DM_ERROR_NOT_ROOT,
    BD_DM_ERROR_TASK,
    BD_DM_ERROR_RAID_FAIL,
    BD_DM_ERROR_RAID_NO_DEVS,
} BDDMError;

/**
 * bd_dm_create_linear:
 * @map_name: name of the map
 * @device: device to create map for
 * @length: length of the mapping in sectors
 * @uuid: (allow-none): UUID for the new dev mapper device or %NULL if not specified
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the new linear mapping @map_name was successfully created
 * for the @device or not
 */
gboolean  bd_dm_create_linear (gchar *map_name, gchar *device, guint64 length, gchar *uuid, GError **error);


/**
 * bd_dm_remove:
 * @map_name: name of the map to remove
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the @map_name map was successfully removed or not
 */
gboolean  bd_dm_remove (gchar *map_name, GError **error);


/**
 * bd_dm_name_from_node:
 * @dm_node: name of the DM node (e.g. "dm-0")
 * @error: (out): place to store error (if any)
 *
 * Returns: map name of the map providing the @dm_node device or %NULL
 * (@error) contains the error in such cases)
 */
gchar* bd_dm_name_from_node (gchar *dm_node, GError **error);


/**
 * bd_dm_node_from_name:
 * @map_name: name of the queried DM map
 * @error: (out): place to store error (if any)
 *
 * Returns: DM node name for the @map_name map or %NULL (@error) contains
 * the error in such cases)
 */
gchar* bd_dm_node_from_name (gchar *map_name, GError **error);


/**
 * bd_dm_map_exists:
 * @map_name: name of the queried map
 * @live_only: whether to go through the live maps only or not
 * @active_only: whether to ignore suspended maps or not
 * @error: (out): place to store error (if any)
 *
 * Returns: whether the given @map_name exists (and is live if @live_only is
 * %TRUE (and is active if @active_only is %TRUE)).
 */
gboolean  bd_dm_map_exists (gchar *map_name, gboolean live_only, gboolean active_only, GError **error);


/**
 * bd_dm_get_member_raid_sets:
 * @name: (allow-none): name of the member
 * @uuid: (allow-none): uuid of the member
 * @major: major number of the device or -1 if not specified
 * @minor: minor number of the device or -1 if not specified
 * @error: (out): variable to store error (if any)
 *
 * Returns: (transfer full) (array zero-terminated=1): list of names of the RAID sets related to
 * the member or %NULL in case of error
 *
 * One of @name, @uuid or @major:@minor has to be given.
 */
gchar** bd_dm_get_member_raid_sets (gchar *name, gchar *uuid, gint major, gint minor, GError **error);


/**
 * bd_dm_activate_raid_set:
 * @name: name of the DM RAID set to activate
 * @error: (out): variable to store error (if any)
 *
 * Returns: whether the RAID set @name was successfully activate or not
 */
gboolean  bd_dm_activate_raid_set (gchar *name, GError **error);


/**
 * bd_dm_deactivate_raid_set:
 * @name: name of the DM RAID set to deactivate
 * @error: (out): variable to store error (if any)
 *
 * Returns: whether the RAID set @name was successfully deactivate or not
 */
gboolean  bd_dm_deactivate_raid_set (gchar *name, GError **error);


/**
 * bd_dm_get_raid_set_type:
 * @name: name of the DM RAID set to get the type of
 * @error: (out): variable to store error (if any)
 *
 * Returns: string representation of the @name RAID set's type
 */
gchar* bd_dm_get_raid_set_type (gchar *name, GError **error);

