﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using VdcCommon.BusinessEntities;
using System.Xml;

namespace VdcCommon.Ovf
{
    public class OvfTemplateReader : OvfReader
    {
        protected VmTemplate _vmTemplate;

        public OvfTemplateReader(XmlDocument document, VmTemplate vmTemplate, List<DiskImage> images)
            : base(document, images)
        {
            _vmTemplate = vmTemplate;
        }

        protected override void ReadOsSection(XmlNode section)
        {
            _vmTemplate.vmt_guid = new Guid(section.Attributes["ovf:id"].Value);
            _vmTemplate.os = (VmOsType)Enum.Parse(typeof(VmOsType), section.SelectSingleNode("Description").InnerText);
        }

        protected override void ReadHardwareSection(XmlNode section)
        {
            XmlNodeList list = section.SelectNodes("Item");
            foreach (XmlNode node in list)
            {
                int resourceType = Convert.ToInt32(node.SelectSingleNode("rasd:ResourceType", _xmlNS).InnerText);

                switch (resourceType)
                {
                    // CPU
                    case 3:
                        _vmTemplate.num_of_sockets = Convert.ToInt32(node.SelectSingleNode("rasd:num_of_sockets", _xmlNS).InnerText);
                        _vmTemplate.cpu_per_socket = Convert.ToInt32(node.SelectSingleNode("rasd:cpu_per_socket", _xmlNS).InnerText);
                        break;

                    // Memory
                    case 4:
                        _vmTemplate.mem_size_mb = Convert.ToInt32(node.SelectSingleNode("rasd:VirtualQuantity", _xmlNS).InnerText);
                        break;

                    // Image
                    case 17:
                        Guid guid = new Guid(node.SelectSingleNode("rasd:InstanceId", _xmlNS).InnerText);
                        DiskImage image = _images.FirstOrDefault(img => img.image_guid == guid);
                        string drive = node.SelectSingleNode("rasd:Caption", _xmlNS).InnerText;
                        if (drive.StartsWith("Drive "))
                        {
                            image.internal_drive_mapping = drive.Substring(6);
                        }
                        image.image_group_id = OvfParser.GetImageGrupIdFromImageFile(node.SelectSingleNode("rasd:HostResource", _xmlNS).InnerText);
                        if (!String.IsNullOrEmpty(node.SelectSingleNode("rasd:Parent", _xmlNS).InnerText))
                        {
                            image.ParentId = new Guid(node.SelectSingleNode("rasd:Parent", _xmlNS).InnerText);
                        }
                        if (!String.IsNullOrEmpty(node.SelectSingleNode("rasd:Template", _xmlNS).InnerText))
                        {
                            image.it_guid = new Guid(node.SelectSingleNode("rasd:Template", _xmlNS).InnerText);
                        }
                        image.appList = node.SelectSingleNode("rasd:ApplicationList", _xmlNS).InnerText;
                        if (!String.IsNullOrEmpty(node.SelectSingleNode("rasd:StorageId", _xmlNS).InnerText))
                        {
                            image.storage_id = new Guid(node.SelectSingleNode("rasd:StorageId", _xmlNS).InnerText);
                        }
                        if (!String.IsNullOrEmpty(node.SelectSingleNode("rasd:StoragePoolId", _xmlNS).InnerText))
                        {
                            image.storage_pool_id = new Guid(node.SelectSingleNode("rasd:StoragePoolId", _xmlNS).InnerText);
                        }
                        DateTime creationDate;
                        if (OvfParser.UtcDateStringToLocaDate(node.SelectSingleNode("rasd:CreationDate", _xmlNS).InnerText, out creationDate))
                        {
                            image.creation_date = creationDate;
                        }
                        DateTime lastModified;
                        if (OvfParser.UtcDateStringToLocaDate(node.SelectSingleNode("rasd:LastModified", _xmlNS).InnerText, out lastModified))
                        {
                            image.lastModified = lastModified;
                        }
                        break;

                    // Network
                    case 10:
                        Interface iface = new Interface();
                        if (!String.IsNullOrEmpty(node.SelectSingleNode("rasd:ResourceSubType", _xmlNS).InnerText))
                        {
                            iface.type = Int32.Parse(node.SelectSingleNode("rasd:ResourceSubType", _xmlNS).InnerText);
                        }
                        iface.network_name = node.SelectSingleNode("rasd:Connection", _xmlNS).InnerText;
                        iface.name = node.SelectSingleNode("rasd:Name", _xmlNS).InnerText;
                        //ITAMAR _vmTemplate.Interfaces.Add(iface);
                        break;

                    // USB
                    case 23:
                        _vmTemplate.usb_policy = (UsbPolicy)Enum.Parse(typeof(UsbPolicy), node.SelectSingleNode("rasd:UsbPolicy", _xmlNS).InnerText);
                        break;

                    // Monitor
                    case 20:
                        _vmTemplate.num_of_monitors = Int32.Parse(node.SelectSingleNode("rasd:VirtualQuantity", _xmlNS).InnerText);
                        break;
                }
            }
        }

        protected override void ReadGeneralData()
        {
            // General Vm
            XmlNode content = _document.SelectSingleNode("//*/Content");

            XmlNode node = content.SelectSingleNode("Name");
            if (node != null)
            {
                _vmTemplate.name = node.InnerText;
            }
            node = content.SelectSingleNode("TemplateId");
            if (node != null)
            {
                if (!string.IsNullOrEmpty(node.InnerText))
                {
                    _vmTemplate.vmt_guid = new Guid(node.InnerText);
                }
            }
            node = content.SelectSingleNode("Description");
            if (node != null)
            {
                _vmTemplate.description = node.InnerText;
            }
            node = content.SelectSingleNode("Domain");
            if (node != null)
            {
                _vmTemplate.domain = node.InnerText;
            }
            node = content.SelectSingleNode("CreationDate");
            DateTime creationDate;
            if (node != null && OvfParser.UtcDateStringToLocaDate(node.InnerText, out creationDate))
            {
                _vmTemplate.creation_date = creationDate;
            }
            node = content.SelectSingleNode("IsAutoSuspend");
            if (node != null)
            {
                _vmTemplate.is_auto_suspend = bool.Parse(node.InnerText);
            }
            node = content.SelectSingleNode("TimeZone");
            if (node != null)
            {
                _vmTemplate.time_zone = node.InnerText;
            }
            node = content.SelectSingleNode("VmType");
            if (node != null)
            {
                if (!string.IsNullOrEmpty(node.InnerText))
                {
                    _vmTemplate.vm_type = (VmType)Convert.ToInt32(node.InnerText);
                }
            }

            XmlNodeList list = content.SelectNodes("Section");
            foreach (XmlNode section in list)
            {
                string value = section.Attributes["xsi:type"].Value;

                switch (value)
                {
                    case "ovf:OperatingSystemSection_Type":
                        ReadOsSection(section);
                        break;

                    case "ovf:VirtualHardwareSection_Type":
                        ReadHardwareSection(section);
                        break;
                }
            }
            node = content.SelectSingleNode("default_display_type");
            if (node != null)
            {
                if (!string.IsNullOrEmpty(node.InnerText))
                {
                    _vmTemplate.default_display_type = (DisplayType)Convert.ToInt32(node.InnerText);
                }
            }
        }
    }
}
