#!/bin/sh

die () {
    printf >&2 "$@"
    exit 1
}

usage () {
    DATE=`date --date "now -1 days" +"%y%m%d%H%M%S%z"`
    echo "Usage:"
    echo "  $0 [installer mode = 0 ] [SVN CA dir path] [Working directory]"
    echo "  $0 [installer mode = 1 ] [Subject] [Country] [Organization] [Alias] [Password] [ANSI Start Date] [Working Directory] [CA Subject]"
    echo "e.g.:"
    echo "  $0 0 /home/user/rhevm/ca /tmp/ca"
    echo "  $0 1 hostname.fqdn US RedHat rhevm mypass $DATE"

    exit 1
}

# Check Args
[ "$#" -ge 3 ] || usage

if [ "$1" -eq 1 ]
then
    # Installer mode
    INST_MODE=1
    SUBJECT=$2
    COUNTRY=$3
    ORG=$4
    ALIAS=$5
    PASS=$6
    DATE=$7
    WORKDIR=$8
    CA_SUBJECT=$9
    [ -d "$8" ] || die "Directory $8 does not exists"
else
    # Developer mode
    INST_MODE=0
    PASS=mypass
    ALIAS=rhevm
    DATE=`date --date "now -1 days" +"%y%m%d%H%M%S%z"`
    SUBJECT=`hostname`
    START_DIR=`pwd`
    COUNTRY=US
    ORG=RedHat
    WORKDIR=$3/ca
    CA_SUBJECT=$SUBJECT
    [ -d "$2" ] || die "Directory $2 does not exist"
    [ -d "$2/keys" ] || die "Directory $2 is not CA !"
    [ -d "$3" ] || die "Directory $3 does not exist"
fi

# Define pubkey2ssh location
if [ -s "$START_DIR/../../3rdparty/pub2ssh/bin/pubkey2ssh" ]; then
	PUB_KEY=$START_DIR/../../3rdparty/pub2ssh/bin
elif [ -s "$WORKDIR/pubkey2ssh" ]; then
	PUB_KEY=$WORKDIR/
else
	die "Unable to find pubkey2ssh executable. Please make sure it's in path."
fi

if [ $INST_MODE -eq 0 ]
then
    echo "# Generating CA in working directory..."
    cp -a $2 $3/
fi

echo " "
echo "} Creating CA..."

# Move to scripts location
cd $WORKDIR

# Create CA
./CreateCA.sh $COUNTRY "$ORG" "CA-$CA_SUBJECT" "$DATE"
[ -s private/ca.pem ] || die "file private/ca.pem does not exist!"
[ -s requests/ca.csr ] || die "file requests/ca.csr does not exist!"
[ -s ca.pem ] || die "file ca.pem does not exist!"

# Create KeyStore
echo " "
echo "} Creating KeyStore..."
./CreateKStore.sh $WORKDIR $COUNTRY "$ORG" "$SUBJECT" $PASS $PASS
[ -s ./.keystore ] || die "file ./.keystore does not exist!"

# Convert pem to der
echo " "
echo "}} Converting formats..."
openssl x509 -in ca.pem -out certs/ca.der -outform DER

# Import CA into keystore 
echo " "
echo "> Importing CA certificate..."
# Generate truststore
keytool -import -noprompt -trustcacerts -alias cacert -keypass $PASS -file certs/ca.der -keystore ./.truststore -storepass $PASS
# Import to keystore as well
./importToKeyStore.sh $WORKDIR/.keystore cacert certs/ca.der $PASS

echo " "
echo "} Creating client certificate for oVirt..."

# Create certificate request
echo " "
echo "}} Creating certificate request..."
./CreateReq.sh 1 $COUNTRY "$ORG" "$SUBJECT" rhevm $PASS $ALIAS

# Sign request
echo " "
echo "}} Signing certificate request..."
./SignReq.sh rhevm.req rhevm.cer 1800 `pwd` "$DATE" $PASS
[ -s certs/rhevm.cer ] || die "file certs/rhevm.cer does not exist!"

echo " "
echo "}} Converting formats..."
openssl x509 -in certs/rhevm.cer -out certs/rhevm.der -outform DER

# Import oVirt certificate into keystore
echo " "
echo "} Importing oVirt certificate..."
./importToKeyStore.sh $WORKDIR/.keystore $ALIAS certs/rhevm.der $PASS

# Export oVirt key as ssh key
echo " "
echo "} Exporting oVirt key as SSH..."
./exportK2SSH.sh $WORKDIR/.keystore $ALIAS rhevm.pub $PASS
$PUB_KEY/pubkey2ssh $WORKDIR/keys/rhevm.pub $ALIAS > $WORKDIR/keys/rhevm.ssh.key.txt
rm -f $WORKDIR/keys/rhevm.pub

if [ $INST_MODE -eq 0 ]
then
    echo " "
    echo "CA Installation Done."
    echo " "
    echo "======================================================"
    echo "Please note:"
    echo "======================================================"
    echo "}}} Do not forget to set:"
    echo " "
    echo "}}} 1. Update ConfigValues.java (if not possible in DB):"
    echo "}}} 1.1. CABaseDirectory to $WORKDIR"
    echo "}}} 1.2. keystoreUrl to $WORKDIR/.keystore"
    echo "}}} 1.3. TruststoreUrl to $WORKDIR/.keystore"
    echo "}}} 1.4. Fix SignScriptName to sh(/bat)"
    echo "}}} 1.5. PublicURLPath to [jboss dir]/jboss-as/server/default/deploy/ROOT.war/"
    echo " "
    echo "}}} 2. DB updates (vdc_options table):"
    echo "}}} 2.1. CertificateFileName $WORKDIR/certs/rhevm.cer"
    echo "}}} 2.2. CABaseDirectory to $WORKDIR"
    echo "}}} 2.3. CARhevmKey to $WORKDIR/private/ca.pem"
    echo " "
    echo "}}} 3. File updates:"
    echo "}}} 3.1. Copy $WORKDIR/keys/rhevm.ssh.key.txt to root context of JBoss."
    echo "}}}      ie: cp $WORKDIR/keys/rhevm.ssh.key.txt /usr/local/jboss-eap-5.0/jboss-as/server/default/deploy/ROOT.war/"
    echo "======================================================"
    echo " "
fi

exit 0

