using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Text;
using System.Threading;
using org.ovirt.engine.ui.uicompat;
using org.ovirt.engine.ui.uicommon.validation;

namespace org.ovirt.engine.ui.uicommon.models
{
	public class ListModel : EntityModel
	{
		#region Events

		public static EventDefinition SelectedItemChangedEventDefinition;
		public Event SelectedItemChangedEvent { get; private set; }

		public static EventDefinition SelectedItemsChangedEventDefinition;
		public Event SelectedItemsChangedEvent { get; private set; }

		public static EventDefinition ItemsChangedEventDefinition;
		public Event ItemsChangedEvent { get; private set; }

		#endregion

		#region Properties

		private IList selectedItems;
		public IList SelectedItems
		{
			get { return selectedItems; }
			set
			{
				if (selectedItems != value)
				{
					SelectedItemsChanging(value, selectedItems);
					selectedItems = value;
					SelectedItemsChanged();
					SelectedItemsChangedEvent.raise(this, EventArgs.Empty);
					OnPropertyChanged(new PropertyChangedEventArgs("SelectedItems"));
				}
			}
		}

		private object selectedItem;
		public object SelectedItem
		{
			get { return selectedItem; }
			set
			{
				if (selectedItem != value)
				{
					OnSelectedItemChanging(value, selectedItem);
					selectedItem = value;
					OnSelectedItemChanged();
					SelectedItemChangedEvent.raise(this, EventArgs.Empty);
					OnPropertyChanged(new PropertyChangedEventArgs("SelectedItem"));
				}
			}
		}

		private IEnumerable items;
		public IEnumerable Items
		{
			get { return items; }
			set
			{
				if (items != value)
				{
					ItemsChanging(value, items);
					items = value;
					ItemsChanged();
					ItemsChangedEvent.raise(this, EventArgs.Empty);
					OnPropertyChanged(new PropertyChangedEventArgs("Items"));
				}
			}
		}

		private bool isEmpty;
		/// <summary>
		/// Gets or sets the value indicating whether this model is empty.
		/// Notice, that this value is not updated automatically.
		/// </summary>
		public bool IsEmpty
		{
			get { return isEmpty; }
			set
			{
				if (isEmpty != value)
				{
					isEmpty = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsEmpty"));
				}
			}
		}

		/// <summary>
		/// Override this property and return true in order to receive
		/// property change notifications for any item but not only for
		/// selected ones.
		/// Pay attention, when property change occurs either SelectedItemPropertyChanged
		/// or ItemPropertyChanged will be called but not both of them.
		/// </summary>
		protected virtual bool NotifyPropertyChangeForAnyItem
		{
			get { return false; }
		}

		#endregion

		static ListModel()
		{
			SelectedItemChangedEventDefinition = new EventDefinition("SelectedItemChanged", typeof(ListModel));
			SelectedItemsChangedEventDefinition = new EventDefinition("SelectedItemsChanged", typeof(ListModel));
			ItemsChangedEventDefinition = new EventDefinition("ItemsChanged", typeof(ListModel));
		}

		public ListModel()
		{
			SelectedItemChangedEvent = new Event(SelectedItemChangedEventDefinition);
			SelectedItemsChangedEvent = new Event(SelectedItemsChangedEventDefinition);
			ItemsChangedEvent = new Event(ItemsChangedEventDefinition);
		}

		protected virtual void OnSelectedItemChanging(object newValue, object oldValue)
		{
		}

		protected virtual void OnSelectedItemChanged()
		{
		}

		protected virtual void SelectedItemsChanged()
		{
		}

		protected virtual void SelectedItemsChanging(IList newValue, IList oldValue)
		{
			//Skip this method when notifying on property change for any
			//item but not only for selected ones is requested.
			//Subscribtion to the event will be done in ItemsCollectionChanged method.
			if (NotifyPropertyChangeForAnyItem)
			{
				return;
			}

			UnsubscribeList(oldValue);
			SubscribeList(newValue);
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(ProvidePropertyChangedEvent.Definition))
			{
				if (NotifyPropertyChangeForAnyItem)
				{
					//If notification on property change for any item was requested,
					//check whether the event was sent by a selected item or not.
					bool anyOfSelectedItem = false;
					if (SelectedItems != null)
					{
						foreach (object item in SelectedItems)
						{
							if (item == sender)
							{
								anyOfSelectedItem = true;
								break;
							}
						}
					}

					if (anyOfSelectedItem)
					{
						SelectedItemPropertyChanged(sender, (PropertyChangedEventArgs)args);
					}
					else
					{
						ItemPropertyChanged(sender, (PropertyChangedEventArgs)args);
					}
				}
				else
				{
					//In this case a sender always will be a one of selected item.
					SelectedItemPropertyChanged(sender, (PropertyChangedEventArgs)args);
				}
			}
			else if (ev.Equals(ProvideCollectionChangedEvent.Definition))
			{
				ItemsCollectionChanged(sender, (NotifyCollectionChangedEventArgs)args);
			}
		}

		/// <summary>
		/// Invoked whenever some property of any selected item was changed.
		/// </summary>
		protected virtual void SelectedItemPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
		}

		/// <summary>
		/// Invoked whenever some property of any item was changed.
		/// For performance considerations, in order to get this method called,
		/// override NotifyPropertyChangeForAnyItem property and return true.
		/// </summary>
		protected virtual void ItemPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
		}

		protected virtual void ItemsChanged()
		{
			// if Items are updated, SelectedItem and SelectedItems become irrelevant:
			SelectedItem = null;
			SelectedItems = null;
		}

		protected virtual void ItemsChanging(IEnumerable newValue, IEnumerable oldValue)
		{
			IProvideCollectionChangedEvent notifier = oldValue as IProvideCollectionChangedEvent;
			if (notifier != null)
			{
				notifier.CollectionChangedEvent.removeListener(this);
			}

			notifier = newValue as IProvideCollectionChangedEvent;
			if (notifier != null)
			{
				notifier.CollectionChangedEvent.addListener(this);
			}


			//Unsure subscribing to the property change notification for all items.
			UnsubscribeList(oldValue);
			SubscribeList(newValue);
		}

		/// <summary>
		/// Invoked whenever items collection was changed, i.e. some items was added or removed.
		/// </summary>
		protected virtual void ItemsCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
		{
			if (!NotifyPropertyChangeForAnyItem)
			{
				return;
			}

			//Track property change on all items as necessary.
			UnsubscribeList(e.OldItems);
			SubscribeList(e.NewItems);
		}


		public void ValidateSelectedItem(IValidation[] validations)
		{
			IsValid = true;

			if (!IsAvailable || !IsChangable)
			{
				return;
			}

			foreach (IValidation validation in validations)
			{
				ValidationResult result = validation.Validate(SelectedItem);
				if (!result.Success)
				{
					foreach (string reason in result.Reasons)
					{
						InvalidityReasons.Add(reason);
					}
					IsValid = false;

					break;
				}
			}
		}

		private void SubscribeList(IEnumerable list)
		{
			if (list == null)
			{
				return;
			}

			foreach (object a in list)
			{
				IProvidePropertyChangedEvent notifier = a as IProvidePropertyChangedEvent;
				if (notifier != null)
				{
					notifier.PropertyChangedEvent.addListener(this);
				}
			}
		}

		private void UnsubscribeList(IEnumerable list)
		{
			if (list == null)
			{
				return;
			}

			foreach (object a in list)
			{
				IProvidePropertyChangedEvent notifier = a as IProvidePropertyChangedEvent;
				if (notifier != null)
				{
					notifier.PropertyChangedEvent.removeListener(this);
				}
			}
		}
	}
}
