using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.dataprovider;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using System.Text.RegularExpressions;
using VdcCommon.VdcQueries;
using org.ovirt.engine.ui.uicompat;

namespace org.ovirt.engine.ui.uicommon.models
{
	/// <summary>
	/// Represents a list model with ability to fetch items both sync and async.
	/// </summary>
	public abstract class SearchableListModel : ListModel
	{
		#region Consts

		private const int UnknownInteger = -1;

		private const string PAGE_STRING_REGEX = @"[\s]+page[\s]+[1-9]+[0-9]*[\s]*$";
		private const string PAGE_NUMBER_REGEX = @"[1-9]+[0-9]*$";

		#endregion

		#region Commands

		public UICommand SearchCommand { get; private set; }
		public UICommand SearchNextPageCommand { get; private set; }
		public UICommand SearchPreviousPageCommand { get; private set; }

		#endregion

		#region Properties

		// Update IsAsync wisely! Set it once after initializing the SearchableListModel object.
		internal bool IsAsync { get; set; }
		public string DefaultSearchString { get; set; }
		public int SearchPageSize { get; set; }

		private RegistrationResult asyncResult;
		public RegistrationResult AsyncResult
		{
			get { return asyncResult; }
			set
			{
				if (asyncResult != value)
				{
					AsyncResultChanging(value, asyncResult);
					asyncResult = value;
					OnPropertyChanged(new PropertyChangedEventArgs("AsyncResult"));
				}
			}
		}

		private string searchString;
		public string SearchString
		{
			get { return searchString; }
			set
			{
				if (searchString != value)
				{
					searchString = value;
					SearchStringChanged();
					OnPropertyChanged(new PropertyChangedEventArgs("SearchString"));
				}
			}
		}

		public int SearchPageNumber
		{
			get
			{
				if (string.IsNullOrEmpty(SearchString))
				{
					return 1;
				}

				// try getting the end of SearchString in the form of "page <n>"
				string pageStringRegex = PAGE_STRING_REGEX;
				
				Match match = Regex.Match(SearchString, pageStringRegex, RegexOptions.IgnoreCase);
				if (match.Success)
				{
					// retrieve the page number itself:
					string pageString = match.Value; // == "page <n>"
					string pageNumberRegex = PAGE_NUMBER_REGEX;
					match = Regex.Match(pageString, pageNumberRegex);
					if (match.Success)
					{
						int retValue;
						if (int.TryParse(match.Value, out retValue))
						{
							return retValue;
						}
					}
				}
				
				return 1;
			}
		}

		public int NextSearchPageNumber
		{
			get
			{
				return SearchPageNumber + 1;
			}
		}

		public int PreviousSearchPageNumber
		{
			get
			{
				return SearchPageNumber == 1 ? 1 : SearchPageNumber - 1;
			}
		}

		#endregion

		private readonly PrivateAsyncCallback asyncCallback;

		protected SearchableListModel()
		{
			//Configure this instance.
			Configurator.Configure(this);

			SearchCommand = new UICommand("Search", this);
			SearchNextPageCommand = new UICommand("SearchNextPage", this);
			SearchPreviousPageCommand = new UICommand("SearchPreviousPage", this);

			SearchPageSize = UnknownInteger;
			asyncCallback = new PrivateAsyncCallback(this);


			// Most of SearchableListModels will not have paging. The ones that
			// should have paging will set it explicitly in their constructors.
			SearchNextPageCommand.IsAvailable = false;
			SearchPreviousPageCommand.IsAvailable = false;
		}

		/// <summary>
		/// Returns value indicating whether the specified search string is
		/// matching this list model.
		/// </summary>
		public virtual bool IsSearchStringMatch(string searchString)
		{
			return true;
		}

		protected virtual void SearchStringChanged()
		{
		}

		public virtual void Search()
		{
			//Defer search if there max result limit was not yet retrieved.
			if (SearchPageSize == UnknownInteger)
			{
				asyncCallback.RequestSearch();
			}
			else
			{
				EnsureAsyncSearchStopped();

				SelectedItem = null;
				SelectedItems = null;

				if (IsAsync)
				{
					AsyncSearch();
				}
				else
				{
					SyncSearch();
				}
			}
		}

		private void AsyncResultChanging(RegistrationResult newValue, RegistrationResult oldValue)
		{
			if (oldValue != null)
			{
				oldValue.RetrievedEvent.removeListener(this);
			}

			if (newValue != null)
			{
				newValue.RetrievedEvent.addListener(this);
			}
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(RegistrationResult.RetrievedEventDefinition))
			{
				AsyncResult_Retrieved();
			}
		}

		private void AsyncResult_Retrieved()
		{
			//Update IsEmpty flag.

			// Note: Do NOT use IList. 'Items' is not necissarily IList 
			// (e.g in Monitor models, the different ListModels' Items are 
			// of type 'valueObjectEnumerableList', which is not IList).
			if (Items != null)
			{
				IEnumerator enumerator = Items.GetEnumerator();
				IsEmpty = enumerator.MoveNext() ? false : true;
			}
			else
			{
				IsEmpty = true;
			}
		}

		private void ResetIsEmpty()
		{
			// Note: Do NOT use IList: 'Items' is not necissarily IList 
			// (e.g in Monitor models, the different ListModels' Items are 
			// of type 'valueObjectEnumerableList', which is not IList).
			if (Items != null)
			{
				IEnumerator enumerator = Items.GetEnumerator();
				if (enumerator.MoveNext())
				{
					IsEmpty = false;
				}
			}
		}

		protected override void ItemsChanged()
		{
			base.ItemsChanged();

			ResetIsEmpty();
			UpdatePagingAvailability();
		}

		protected override void ItemsCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
		{
			base.ItemsCollectionChanged(sender, e);

			ResetIsEmpty();
			UpdatePagingAvailability();
		}

		protected void UpdatePagingAvailability()
		{
			SearchNextPageCommand.IsExecutionAllowed =
				SearchNextPageCommand.IsAvailable && NextSearchPageAllowed;
			SearchPreviousPageCommand.IsExecutionAllowed =
				SearchPreviousPageCommand.IsAvailable && PreviousSearchPageAllowed;
		}

		private void SetSearchStringPage(int newSearchPageNumber)
		{
			if (Regex.IsMatch(SearchString, PAGE_STRING_REGEX, RegexOptions.IgnoreCase))
			{
				SearchString =
					Regex.Replace(SearchString, PAGE_STRING_REGEX, string.Format(" page {0}", newSearchPageNumber));
			}
			else
			{
				SearchString = string.Format("{0} page {1}", SearchString, newSearchPageNumber);
			}
		}

		protected virtual void SearchNextPage()
		{
			SetSearchStringPage(NextSearchPageNumber);
			SearchCommand.Execute();
		}

		protected virtual void SearchPreviousPage()
		{
			SetSearchStringPage(PreviousSearchPageNumber);
			SearchCommand.Execute();
		}

		protected bool NextSearchPageAllowed
		{
			get
			{
				if (!SearchNextPageCommand.IsAvailable || Items == null || IteratorUtils.moveNext(Items.GetEnumerator()) == false)
				{
					return false;
				}

				bool retValue = true;

				// ** TODO: Inefficient performance-wise! If 'Items' was ICollection or IList
				// ** it would be better, since we could simply check its 'Count' property.

				int pageSize = SearchPageSize;

				if (pageSize > 0)
				{
					IEnumerator e = Items.GetEnumerator();
					int itemsCountInCurrentPage = 0;
					while (IteratorUtils.moveNext(e))
					{
						itemsCountInCurrentPage++;
					}

					if (itemsCountInCurrentPage < pageSize)
					{
						// current page contains results quantity smaller than 
						// the pageSize -> there is no next page:
						retValue = false;
					}
				}

				return retValue;
			}
		}

		protected bool PreviousSearchPageAllowed
		{
			get
			{
				return SearchPreviousPageCommand.IsAvailable && SearchPageNumber > 1;
			}
		}

		/// <summary>
		/// Override this method to take care on sync fetching.
		/// </summary>
		protected virtual void SyncSearch()
		{
		}

		public void SyncSearch(VdcQueryType vdcQueryType, VdcQueryParametersBase vdcQueryParametersBase)
		{
			AsyncQuery _asyncQuery = new AsyncQuery();
			_asyncQuery.Model = this;
			/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object ReturnValue)
			{
				SearchableListModel searchableListModel = (SearchableListModel)model;
				searchableListModel.Items = (IEnumerable)((VdcQueryReturnValue)ReturnValue).ReturnValue;
			};//END_DELEGATE

			Frontend.RunQuery(vdcQueryType,
				vdcQueryParametersBase,
				_asyncQuery);
		}

		/// <summary>
		/// Override this method to take care on async fetching.
		/// </summary>
		protected virtual void AsyncSearch()
		{
		}

		internal virtual void EnsureAsyncSearchStopped()
		{
			if (AsyncResult != null && !AsyncResult.Id.Equals(Guid.Empty))
			{
				Frontend.UnregisterQuery(AsyncResult.Id);
				AsyncResult = null;
			}
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == SearchCommand)
			{
				Search();
			}
			else if (command == SearchNextPageCommand)
			{
				SearchNextPage();
			}
			else if (command == SearchPreviousPageCommand)
			{
				SearchPreviousPage();
			}
		}

		#region Async Callback

		public sealed class PrivateAsyncCallback
		{
			private readonly SearchableListModel model;
			private bool searchRequested;


			public PrivateAsyncCallback(SearchableListModel model)
			{
				this.model = model;
				AsyncQuery _asyncQuery1 = new AsyncQuery();
				_asyncQuery1.Model = this;
				/*START_DELEGATE*/_asyncQuery1.asyncCallback = delegate(Object model1, Object result1)
				                            	{
													PrivateAsyncCallback privateAsyncCallback1 = (PrivateAsyncCallback)model1;
													privateAsyncCallback1.ApplySearchPageSize((int)result1);
				};//END_DELEGATE
				AsyncDataProvider.GetSearchResultsLimit(_asyncQuery1);
			}

			public void RequestSearch()
			{
				searchRequested = true;
			}

			private void ApplySearchPageSize(int value)
			{
				model.SearchPageSize = value;

				//If there search was requested before max result limit was retrieved, do it now.
				if (searchRequested)
				{
					model.SearchCommand.Execute();
				}

				//Sure paging functionality.
				model.UpdatePagingAvailability();
			}
		}

		#endregion
	}
}
