using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Text;
using System.Windows.Input;
using org.ovirt.engine.ui.uicommon.models.configure;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;

namespace org.ovirt.engine.ui.uicommon.models.clusters
{
	public class ClusterListModel : ListWithDetailsModel, ISupportSystemTreeContext
	{
		#region Commands

		public UICommand NewCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }
		public UICommand GuideCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		private Model confirmWindow;
		public Model ConfirmWindow
		{
			get { return confirmWindow; }
			set
			{
				if (confirmWindow != value)
				{
					confirmWindow = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ConfirmWindow"));
				}
			}
		}

		protected object[] SelectedKeys
		{
			//			get { return SelectedItems == null ? new object[0] : SelectedItems.Cast<VDSGroup>().Select(a => a.ID).Cast<object>().ToArray(); }
			get
			{
				if (SelectedItems == null)
				{
					return new object[0];
				}
				else
				{
					List<object> items = new List<object>();
					foreach (object i in SelectedItems)
					{
						items.Add(((VDSGroup)i).ID);
					}
					return items.ToArray();
				}
			}
		}

		public object GuideContext { get; set; }

		#endregion


		public ClusterListModel()
		{
			Title = "Clusters";

			DefaultSearchString = "Cluster:";
			SearchString = DefaultSearchString;

			NewCommand = new UICommand("New", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);
			GuideCommand = new UICommand("Guide", this);

			UpdateActionAvailability();

			SearchNextPageCommand.IsAvailable = true;
			SearchPreviousPageCommand.IsAvailable = true;
		}

		public void Guide()
		{
			ClusterGuideModel model = new ClusterGuideModel();
			Window = model;
			model.Title = "New Cluster - Guide Me";
			model.HashName = "new_cluster_-_guide_me";

			model.Entity = GuideContext != null
				? DataProvider.GetClusterById(GuideContext is guid ? (Guid)(guid)GuideContext : (Guid)GuideContext)
				: null;

			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Configure Later",
					IsDefault = true,
					IsCancel = true
				});
		}

		protected override void InitDetailModels()
		{
			base.InitDetailModels();

			ObservableCollection<EntityModel> list = new ObservableCollection<EntityModel>();
			list.Add(new ClusterPolicyModel());
			list.Add(new ClusterHostListModel());
			list.Add(new ClusterVmListModel());
			list.Add(new ClusterNetworkListModel());
			list.Add(new PermissionListModel());
			DetailModels = list;
		}

		public override bool IsSearchStringMatch(string searchString)
		{
			return searchString.Trim().ToLower().StartsWith("cluster");
		}

		protected override void SyncSearch()
		{
			base.SyncSearch(VdcQueryType.Search, new SearchParameters(SearchString, SearchType.Cluster)
			{
				MaxCount = SearchPageSize
			});
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterSearch(SearchString, SearchType.Cluster, SearchPageSize);
			Items = AsyncResult.Data;
		}

		public void New()
		{
			if (Window != null)
			{
				return;
			}

			ClusterModel model = new ClusterModel();
			Window = model;
			model.Title = "New Cluster";
			model.HashName = "new_cluster";
			model.IsNew = true;


			List<storage_pool> dataCenters = DataProvider.GetDataCenterList();
			model.DataCenter.Items = dataCenters;

			//Be aware of system tree selection.
			//Strict data center as neccessary.
			if (SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type != SystemTreeItemType.System)
			{
				storage_pool selectDataCenter = (storage_pool)SystemTreeSelectedItem.Entity;

				model.DataCenter.SelectedItem = Linq.FirstOrDefault(dataCenters, new Linq.DataCenterPredicate(selectDataCenter.Id));
				model.DataCenter.IsChangable = false;
			}
			else
			{
				model.DataCenter.SelectedItem = Linq.FirstOrDefault(dataCenters);
			}


			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void Edit()
		{
			VDSGroup cluster = (VDSGroup)SelectedItem;

			if (Window != null)
			{
				return;
			}

			ClusterModel model = new ClusterModel();
			Window = model;
			model.Title = "Edit Cluster";
			model.HashName = "edit_cluster";
			model.OriginalName = cluster.name;
			model.Name.Entity = cluster.name;

			if (SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.Cluster)
			{
				model.Name.IsChangable = false;
				model.Name.Info = "Cannot edit Cluster's Name in tree context"; ;
			}

			model.Description.Entity = cluster.description;
			model.MemoryOverCommit = cluster.max_vds_memory_over_commit;


			List<storage_pool> storagePools = DataProvider.GetDataCenterList();
			model.DataCenter.Items = storagePools;

			model.DataCenter.SelectedItem = default(storage_pool);
			foreach (storage_pool a in storagePools)
			{
				if (cluster.storage_pool_id != null && a.Id.Equals(cluster.storage_pool_id))
				{
					model.DataCenter.SelectedItem = a;
					break;
				}
			}
			model.DataCenter.IsChangable = model.DataCenter.SelectedItem == null;

			// When editing, the possible version values should be retrieved by the cluster.
			model.Version.Items = DataProvider.GetClusterVersions(cluster.ID);
			model.Version.SelectedItem = (Version)cluster.compatibility_version;

			model.MigrateOnErrorOption = cluster.MigrateOnError;

			model.CPU.SelectedItem = null;
			foreach (ServerCpu a in (List<ServerCpu>)model.CPU.Items)
			{
				if (a.CpuName == cluster.cpu_name)
				{
					model.CPU.SelectedItem = a;
					break;
				}
			}

			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Cluster(s)";
			model.HashName = "remove_cluster";
			model.Message = "Cluster(s)";

			List<string> list = new List<string>();
			foreach (VDSGroup a in Linq.Cast<VDSGroup>(SelectedItems))
			{
				list.Add(a.name);
			}
			model.Items = list;


			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnRemove()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> prms = new List<VdcActionParametersBase>();
			foreach (object a in SelectedItems)
			{
				prms.Add(new VdsGroupParametersBase(((VDSGroup)a).ID));
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.RemoveVdsGroup, prms,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		public void OnSave()
		{
			ClusterModel model = (ClusterModel)Window;
			if (!model.Validate())
			{
				return;
			}

			if (!model.IsNew && (Version)model.Version.SelectedItem != ((VDSGroup)SelectedItem).compatibility_version)
			{
				ConfirmationModel confirmModel = new ConfirmationModel();
				ConfirmWindow = confirmModel;
				confirmModel.Title = "Change Cluster Compatibility Version";
				confirmModel.HashName = "change_cluster_compatibility_version";
				confirmModel.Message = "You are about to change the Cluster Compatibility Version. Are you sure you want to continue?";

				ConfirmWindow.Commands.Add(
					new UICommand("OnSaveInternal", this)
					{
						Title = "OK",
						IsDefault = true
					});
				ConfirmWindow.Commands.Add(
					new UICommand("CancelConfirmation", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
			else
			{
				OnSaveInternal();
			}
		}

		public void OnSaveInternal()
		{
			ClusterModel model = (ClusterModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			//cancel confirm window if there is
			CancelConfirmation();

			VDSGroup cluster = model.IsNew ? new VDSGroup() : (VDSGroup)Cloner.Clone(SelectedItem);

			Version version = (Version)model.Version.SelectedItem;

			cluster.name = (string)model.Name.Entity;
			cluster.description = (string)model.Description.Entity;
			cluster.storage_pool_id = ((storage_pool)model.DataCenter.SelectedItem).Id;
			cluster.cpu_name = ((ServerCpu)model.CPU.SelectedItem).CpuName;
			cluster.max_vds_memory_over_commit = model.MemoryOverCommit;
			cluster.TransparentHugepages = version.CompareTo(new Version("3.0")) >= 0;
			cluster.compatibility_version = version;
			cluster.MigrateOnError = model.MigrateOnErrorOption;


			model.StartProgress(null);

			Frontend.RunAction(model.IsNew ? VdcActionType.AddVdsGroup : VdcActionType.UpdateVdsGroup,
				new VdsGroupOperationParameters(cluster),
				result =>
				{
					ClusterListModel localModel = (ClusterListModel)result.State;

					localModel.PostOnSaveInternal(result.ReturnValue);
				},
				this
			);
		}

		public void PostOnSaveInternal(VdcReturnValueBase returnValue)
		{
			ClusterModel model = (ClusterModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();

				if (model.IsNew)
				{
					GuideContext = returnValue.ActionReturnValue;
					UpdateActionAvailability();
					GuideCommand.Execute();
				}
			}
		}

		public void Cancel()
		{
			CancelConfirmation();

			GuideContext = null;
			Window = null;

			UpdateActionAvailability();
		}

		public void CancelConfirmation()
		{
			ConfirmWindow = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void ItemsCollectionChanged(object sender, NotifyCollectionChangedEventArgs e)
		{
			base.ItemsCollectionChanged(sender, e);

			//Try to select an item corresponding to the system tree selection.
			if (SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.Cluster)
			{
				VDSGroup cluster = (VDSGroup)SystemTreeSelectedItem.Entity;

				SelectedItem = Linq.FirstOrDefault(Linq.Cast<VDSGroup>(Items), new Linq.ClusterPredicate(cluster.ID));
			}
		}

		private void UpdateActionAvailability()
		{
			EditCommand.IsExecutionAllowed = SelectedItem != null
				&& SelectedItems != null
				&& SelectedItems.Count == 1;

			GuideCommand.IsExecutionAllowed = GuideContext != null
				|| (SelectedItem != null
				&& SelectedItems != null
				&& SelectedItems.Count == 1);

			RemoveCommand.IsExecutionAllowed = SelectedItems != null
				&& SelectedItems.Count > 0;


			//System tree dependent actions.
			bool isAvailable = !(SystemTreeSelectedItem != null && SystemTreeSelectedItem.Type == SystemTreeItemType.Cluster);

			NewCommand.IsAvailable = isAvailable;
			RemoveCommand.IsAvailable = isAvailable;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewCommand)
			{
				New();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command == GuideCommand)
			{
				Guide();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if (command.Name == "OnSaveInternal")
			{
				OnSaveInternal();
			}
			else if (command.Name == "CancelConfirmation")
			{
				CancelConfirmation();
			}
		}


		private SystemTreeItemModel systemTreeSelectedItem;
		public SystemTreeItemModel SystemTreeSelectedItem
		{
			get { return systemTreeSelectedItem; }
			set
			{
				if (systemTreeSelectedItem != value)
				{
					systemTreeSelectedItem = value;
					OnSystemTreeSelectedItemChanged();
				}
			}
		}

		private void OnSystemTreeSelectedItemChanged()
		{
			UpdateActionAvailability();
		}
	}
}
