using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;
using org.ovirt.engine.ui.uicompat;
using org.ovirt.engine.ui.uicommon.validation;
using VdcCommon.BusinessEntities;

namespace org.ovirt.engine.ui.uicommon.models.clusters
{
	public class ClusterModel : Model
	{
		#region Properties

		public bool IsNew { get; set; }
		public string OriginalName { get; set; }
		public Guid? ClusterId { get; set; }
		public EntityModel Name { get; set; }
		public EntityModel Description { get; set; }
		public ListModel DataCenter { get; set; }
		public ListModel CPU { get; set; }
		public ListModel Version { get; set; }

		public EntityModel OptimizationNone { get; set; }
		public EntityModel OptimizationForServer { get; set; }
		public EntityModel OptimizationForDesktop { get; set; }
		public EntityModel OptimizationCustom { get; set; }

		bool isGeneralTabValid;
		public bool IsGeneralTabValid
		{
			get { return isGeneralTabValid; }
			set
			{
				if (isGeneralTabValid != value)
				{
					isGeneralTabValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsGeneralTabValid"));
				}
			}
		}

		private MigrateOnErrorOptions migrateOnErrorOption;
		public MigrateOnErrorOptions MigrateOnErrorOption
		{
			get
			{
				return migrateOnErrorOption;
			}

			set
			{
				if (migrateOnErrorOption != value)
				{
					migrateOnErrorOption = value;
					OnPropertyChanged(new PropertyChangedEventArgs("MigrateOnErrorOption"));
				}
			}
		}

		public bool isResiliencePolicyTabAvailable { get; set; }
		public bool IsResiliencePolicyTabAvailable
		{
			get
			{
				return isResiliencePolicyTabAvailable;
			}

			set
			{
				if (isResiliencePolicyTabAvailable != value)
				{
					isResiliencePolicyTabAvailable = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsResiliencePolicyTabAvailable"));
				}
			}
		}

		public int MemoryOverCommit
		{
			get
			{
				if (OptimizationNone.IsSelected)
				{
					return (int)OptimizationNone.Entity;
				}

				if (OptimizationForServer.IsSelected)
				{
					return (int)OptimizationForServer.Entity;
				}

				if (OptimizationForDesktop.IsSelected)
				{
					return (int)OptimizationForDesktop.Entity;
				}

				if (OptimizationCustom.IsSelected)
				{
					return (int)OptimizationCustom.Entity;
				}

				return DataProvider.GetClusterDefaultMemoryOverCommit();
			}
			set
			{
				OptimizationNone.IsSelected = value == (int)OptimizationNone.Entity;
				OptimizationForServer.IsSelected = value == (int)OptimizationForServer.Entity;
				OptimizationForDesktop.IsSelected = value == (int)OptimizationForDesktop.Entity;

				if (!OptimizationNone.IsSelected && !OptimizationForServer.IsSelected && !OptimizationForDesktop.IsSelected)
				{
					OptimizationCustom.IsAvailable = true;
					OptimizationCustom.Entity = value;
					OptimizationCustom.IsSelected = true;
				}
			}
		}

		#endregion

		public ClusterModel()
		{
			Name = new EntityModel();
			Description = new EntityModel();


			//Optimization methods.
			int defaultOverCommit = DataProvider.GetClusterDefaultMemoryOverCommit();
			int serverOverCommit = DataProvider.GetClusterServerMemoryOverCommit();
			int desktopOverCommit = DataProvider.GetClusterDesktopMemoryOverCommit();

			OptimizationNone =
				new EntityModel
				{
					Entity = defaultOverCommit,
					IsSelected = desktopOverCommit != defaultOverCommit && serverOverCommit != defaultOverCommit
				};

			OptimizationForServer =
				new EntityModel
				{
					Entity = serverOverCommit,
					IsSelected = serverOverCommit == defaultOverCommit
				};

			OptimizationForDesktop =
				new EntityModel
				{
					Entity = desktopOverCommit,
					IsSelected = desktopOverCommit == defaultOverCommit
				};

			OptimizationCustom =
				new EntityModel
				{
					IsAvailable = false
				};


			DataCenter = new ListModel();
			DataCenter.SelectedItemChangedEvent.addListener(this);
			CPU = new ListModel();
			Version = new ListModel();
			Version.SelectedItemChangedEvent.addListener(this);
			MigrateOnErrorOption = MigrateOnErrorOptions.YES;

			IsGeneralTabValid = true;
			IsResiliencePolicyTabAvailable = true;
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(ListModel.SelectedItemChangedEventDefinition))
			{
				if (sender == DataCenter)
				{
					StoragePool_SelectedItemChanged(args);
				}
				else if (sender == Version)
				{
					Version_SelectedItemChanged(args);
				}
			}
		}

		void Version_SelectedItemChanged(EventArgs e)
		{
			Version version;
			if (Version.SelectedItem != null)
			{
				version = (Version)Version.SelectedItem;
			}
			else
			{
				version = ((storage_pool)DataCenter.SelectedItem).compatibility_version;
			}

			List<ServerCpu> cpus = DataProvider.GetCPUList(version);
			ServerCpu oldSelectedCpu = (ServerCpu)CPU.SelectedItem;
			CPU.Items = cpus;

			if (oldSelectedCpu != null)
			{
				CPU.SelectedItem = Linq.FirstOrDefault(cpus, new Linq.ServerCpuPredicate(oldSelectedCpu.CpuName));
			}

			if (CPU.SelectedItem == null)
			{
				CPU.SelectedItem = Linq.FirstOrDefault(cpus);
			}
		}

		void StoragePool_SelectedItemChanged(EventArgs e)
		{
			// possible versions for new cluster (when editing cluster, this event won't occur)
			// are actually the possible versions for the data-center that the cluster is going
			// to be attached to.
			storage_pool selectedDataCenter = (storage_pool)DataCenter.SelectedItem;
			if (selectedDataCenter.storage_pool_type == StorageType.LOCALFS)
			{
				IsResiliencePolicyTabAvailable = false;
			}
			else
			{
				IsResiliencePolicyTabAvailable = true;
			}
			List<Version> versions = DataProvider.GetDataCenterClusterVersions(selectedDataCenter == null ? null : (Guid?)(selectedDataCenter.Id));
			Version.Items = versions;
			if (!versions.Contains((Version)Version.SelectedItem))
			{
				if (versions.Contains(selectedDataCenter.compatibility_version))
				{
					Version.SelectedItem = (Version)selectedDataCenter.compatibility_version;
				}
				else
				{
					Version.SelectedItem = Linq.SelectHighestVersion(versions);
				}

			}
		}

		internal bool Validate()
		{
			return Validate(true);
		}

		internal bool Validate(bool validateStoragePool)
		{
			Name.ValidateEntity(new IValidation[] { new NotEmptyValidation(), new NoSpacesValidation(),
					new RegexValidation()
						{
							Expression= "^[A-Za-z0-9_-]+$",
							Message = "Name can contain only 'A-Z', 'a-z', '0-9', '_' or '-' characters."
						}
			});
			if (validateStoragePool)
			{
				DataCenter.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });
			}
			CPU.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });
			Version.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });


			string name = (string)Name.Entity;

			//Check name unicitate.
			if (String.Compare(name, OriginalName, true) != 0 && !DataProvider.IsClusterNameUnique(name))
			{
				Name.IsValid = false;
				Name.InvalidityReasons.Add("Name must be unique.");
			}

			IsGeneralTabValid =
				Name.IsValid
				&& DataCenter.IsValid
				&& CPU.IsValid
				&& Version.IsValid;

			return Name.IsValid
				   && DataCenter.IsValid
				   && CPU.IsValid
				   && Version.IsValid;
		}
	}
}
