using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using System.Windows;
using org.ovirt.engine.ui.uicommon.validation;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using System.Collections;
using System.ComponentModel;

namespace org.ovirt.engine.ui.uicommon.models.clusters
{
	public class ClusterPolicyModel : EntityModel
	{
		private int? lowLimitPowerSaving;
		private int? highLimitPowerSaving;
		private int? highLimitEvenlyDistributed;

		#region Commands

		public UICommand EditCommand { get; private set; }

		#endregion

		#region Properties

		public new VDSGroup Entity
		{
			get { return base.Entity as VDSGroup; }
			set { base.Entity = value; }
		}

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		public EntityModel OverCommitTime { get; set; }

		private bool hasOverCommitLowLevel;
		public bool HasOverCommitLowLevel
		{
			get { return hasOverCommitLowLevel; }
			set
			{
				if (hasOverCommitLowLevel != value)
				{
					hasOverCommitLowLevel = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasOverCommitLowLevel"));
				}
			}
		}

		public bool hasOverCommitHighLevel;
		public bool HasOverCommitHighLevel
		{
			get { return hasOverCommitHighLevel; }
			set
			{
				if (hasOverCommitHighLevel != value)
				{
					hasOverCommitHighLevel = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasOverCommitHighLevel"));
				}
			}
		}
		//Editing features
		private VdsSelectionAlgorithm selectionAlgorithm;
		public VdsSelectionAlgorithm SelectionAlgorithm
		{
			get { return selectionAlgorithm; }
			set
			{
				if (selectionAlgorithm != value)
				{
					selectionAlgorithm = value;
					SelectionAlgorithmChanged();
					OnPropertyChanged(new PropertyChangedEventArgs("SelectionAlgorithm"));
				}
			}
		}

		private int overCommitLowLevel;
		public int OverCommitLowLevel
		{
			get { return overCommitLowLevel; }
			set
			{
				if (overCommitLowLevel != value)
				{
					overCommitLowLevel = value;
					OnPropertyChanged(new PropertyChangedEventArgs("OverCommitLowLevel"));
				}
			}
		}

		private int overCommitHighLevel;
		public int OverCommitHighLevel
		{
			get { return overCommitHighLevel; }
			set
			{
				if (overCommitHighLevel != value)
				{
					overCommitHighLevel = value;
					OnPropertyChanged(new PropertyChangedEventArgs("OverCommitHighLevel"));
				}
			}
		}

		public void SaveDefaultValues()
		{
			if (SelectionAlgorithm == VdsSelectionAlgorithm.EvenlyDistribute)
			{
				highLimitEvenlyDistributed = OverCommitHighLevel;
			}
			else if (SelectionAlgorithm == VdsSelectionAlgorithm.PowerSave)
			{
				lowLimitPowerSaving = OverCommitLowLevel;
				highLimitPowerSaving = OverCommitHighLevel;
			}
		}

		#endregion


		public ClusterPolicyModel()
		{
			Title = "General";

			EditCommand = new UICommand("Edit", this);
			OverCommitTime = new EntityModel();

			// Set all properties according to default selected algorithm:
			SelectionAlgorithmChanged();
		}


		public void Edit()
		{
			if (Window != null)
			{
				return;
			}

			ClusterPolicyModel model = new ClusterPolicyModel();
			Window = model;
			model.Title = "Edit Policy";
			model.HashName = "edit_policy";

			model.SelectionAlgorithm = Entity.selection_algorithm;
			model.OverCommitTime.Entity = Entity.cpu_over_commit_duration_minutes;
			model.OverCommitLowLevel = Entity.low_utilization;
			model.OverCommitHighLevel = Entity.high_utilization;

			model.SaveDefaultValues();

			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnSave()
		{
			ClusterPolicyModel model = (ClusterPolicyModel)Window;

			if (Entity == null)
			{
				Cancel();
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			VDSGroup cluster = (VDSGroup)Cloner.Clone(Entity);
			cluster.selection_algorithm = model.SelectionAlgorithm;
			if (model.OverCommitTime.IsAvailable)
			{
				cluster.cpu_over_commit_duration_minutes = int.Parse(model.OverCommitTime.Entity.ToString());
			}
			cluster.low_utilization = model.OverCommitLowLevel;
			cluster.high_utilization = model.OverCommitHighLevel;

			VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.UpdateVdsGroup,
				new VdsGroupOperationParameters(cluster));

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();
			}
		}

		internal bool Validate()
		{
			OverCommitTime.ValidateEntity(
				new IValidation[]
				{
					new NotEmptyValidation(),
					new IntegerValidation { Minimum = 1, Maximum = 100 }
				});

			return OverCommitTime.IsValid;
		}

		public void Cancel()
		{
			Window = null;
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();

			if (Entity != null)
			{
				UpdateProperties();
			}

			UpdateActionAvailability();
		}

		protected override void EntityPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.EntityPropertyChanged(sender, e);

			if (e.PropertyName == "selection_algorithm")
			{
				UpdateProperties();
			}
		}

		private void UpdateActionAvailability()
		{
			EditCommand.IsExecutionAllowed = Entity != null;
		}

		private void UpdateProperties()
		{
			OverCommitTime.IsAvailable = Entity.selection_algorithm != VdsSelectionAlgorithm.None;
			OverCommitTime.IsChangable = OverCommitTime.IsAvailable;
			HasOverCommitLowLevel = Entity.selection_algorithm == VdsSelectionAlgorithm.PowerSave;
			HasOverCommitHighLevel = Entity.selection_algorithm != VdsSelectionAlgorithm.None;
		}

		private void SelectionAlgorithmChanged()
		{
			HasOverCommitLowLevel = SelectionAlgorithm != VdsSelectionAlgorithm.EvenlyDistribute;

			switch (SelectionAlgorithm)
			{
				case VdsSelectionAlgorithm.None:
					OverCommitTime.IsAvailable = false;
					HasOverCommitLowLevel = false;
					HasOverCommitHighLevel = false;
					OverCommitLowLevel = 0;
					OverCommitHighLevel = 0;
					break;

				case VdsSelectionAlgorithm.EvenlyDistribute:
					OverCommitTime.IsAvailable = true;
					OverCommitTime.IsChangable = true;
					HasOverCommitLowLevel = false;
					HasOverCommitHighLevel = true;
					OverCommitLowLevel = 0;
					if (highLimitEvenlyDistributed == null)
					{
						highLimitEvenlyDistributed = DataProvider.GetHighUtilizationForEvenDistribution();
					}
					OverCommitHighLevel = highLimitEvenlyDistributed.GetValueOrDefault();
					break;

				case VdsSelectionAlgorithm.PowerSave:
					OverCommitTime.IsAvailable = true;
					OverCommitTime.IsChangable = true;
					HasOverCommitLowLevel = true;
					HasOverCommitHighLevel = true;
					if (lowLimitPowerSaving == null)
					{
						lowLimitPowerSaving = DataProvider.GetLowUtilizationForPowerSave();
					}
					OverCommitLowLevel = lowLimitPowerSaving.GetValueOrDefault();

					if (highLimitPowerSaving == null)
					{
						highLimitPowerSaving = DataProvider.GetHighUtilizationForPowerSave();
					}
					OverCommitHighLevel = highLimitPowerSaving.GetValueOrDefault();
					break;
			}
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == EditCommand)
			{
				Edit();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
		}
	}
}
