using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.common;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;

namespace org.ovirt.engine.ui.uicommon.models.datacenters
{
	public class DataCenterNetworkListModel : SearchableListModel, IFrontendMultipleQueryAsyncCallback
	{
		#region Consts
		const string RHEVM_NETWORK = "rhevm";
		#endregion

		#region Commands

		public UICommand NewCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }

		#endregion

		#region Properties

		public new storage_pool Entity
		{
			get { return (storage_pool)base.Entity; }
			set { base.Entity = value; }
		}

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		private Model confirmWindow;
		public Model ConfirmWindow
		{
			get { return confirmWindow; }
			set
			{
				if (confirmWindow != value)
				{
					confirmWindow = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ConfirmWindow"));
				}
			}
		}

		public List<VDSGroup> ClusterList { get; set; }
		public List<SelectionTreeNodeModel> SelectionNodeList { get; set; }

		#endregion

		public DataCenterNetworkListModel()
		{
			Title = "Logical Networks";

			NewCommand = new UICommand("New", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);

			UpdateActionAvailability();
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();
			SearchCommand.Execute();
		}

		public override void Search()
		{
			if (Entity != null)
			{
				base.Search();
			}
		}

		protected override void SyncSearch()
		{
			base.SyncSearch();
			
			AsyncQuery _asyncQuery = new AsyncQuery();
			_asyncQuery.Model = this;
			/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object ReturnValue)
			{
				SearchableListModel searchableListModel = (SearchableListModel)model;
				searchableListModel.Items = (List<network>)((VdcQueryReturnValue)ReturnValue).ReturnValue;
			};//END_DELEGATE

			Frontend.RunQuery(VdcQueryType.GetAllNetworks,
				new GetAllNetworkQueryParamenters(Entity.Id),
				_asyncQuery);
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterQuery(VdcQueryType.GetAllNetworks, new GetAllNetworkQueryParamenters(Entity.Id));
			Items = AsyncResult.Data;
		}

		public void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Logical Network(s)";
			model.HashName = "remove_logical_network";
			model.Message = "Logical Network(s)";

			List<string> list = new List<string>();
			foreach (network a in Linq.Cast<network>(SelectedItems))
			{
				list.Add(a.name);
			}
			model.Items = list;


			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnRemove()
		{
			List<VdcActionParametersBase> pb = new List<VdcActionParametersBase>();
			foreach (network a in Linq.Cast<network>(SelectedItems))
				pb.Add((VdcActionParametersBase)new AddNetworkStoragePoolParameters(Entity.Id, a));
			Frontend.RunMultipleAction(VdcActionType.RemoveNetwork, pb);

			Cancel();
		}

		public void Edit()
		{
			network network = (network)SelectedItem;

			if (Window != null)
			{
				return;
			}

			DataCenterNetworkModel model = new DataCenterNetworkModel();
			Window = model;
			model.Title = "Edit Logical Network";
			model.HashName = "edit_logical_network";
			model.Name.Entity = network.name;
			model.Description.Entity = network.description;
			model.IsStpEnabled = network.stp;
			model.HasVLanTag = network.vlan_id.HasValue;
			model.VLanTag.Entity = network.vlan_id.GetValueOrDefault();

			ClusterList = DataProvider.GetClusterList(Entity.Id);
			SelectionNodeList = new List<SelectionTreeNodeModel>();
			List<VdcQueryParametersBase> parametersList = new List<VdcQueryParametersBase>();
			List<VdcQueryType> queryTypeList = new List<VdcQueryType>();
			foreach (VDSGroup vdsGroup in ClusterList)
			{
				queryTypeList.Add(VdcQueryType.GetAllNetworksByClusterId);
				parametersList.Add(new VdsGroupQueryParamenters(vdsGroup.ID));
				SelectionNodeList.Add(new SelectionTreeNodeModel() { IsSelectedNullable = false, Entity = vdsGroup, Description = vdsGroup.name });
			}
			Frontend.RunMultipleQueries(queryTypeList, parametersList, this);
			model.DetachAllCommand = new UICommand("DetachClusters", this);
			//cannot detach rhevm networks from clusters
			if (network.name == RHEVM_NETWORK)
			{
				foreach (SelectionTreeNodeModel nodeModel in SelectionNodeList)
				{
					nodeModel.IsChangable = false;
				}
				model.DetachAllCommand.IsAvailable = false;
				model.Name.IsChangable = false;
				model.Message = "Cannot detach Management Network from Clusters";
			}
		}

		public void New()
		{
			if (Window != null)
			{
				return;
			}

			DataCenterNetworkModel model = new DataCenterNetworkModel();
			Window = model;
			model.Title = "New Logical Network";
			model.HashName = "new_logical_network";

			model.IsNew = true;
			model.Clusters = DataProvider.GetClusterList(Entity.Id);
			

			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.DetachAllCommand = new UICommand("DetachClusters", this);
			model.DetachAllAvailable.Entity = false;
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnSave()
		{
			DataCenterNetworkModel model = (DataCenterNetworkModel)Window;

			if (Entity == null || (!model.IsNew && SelectedItem == null))
			{
				Cancel();
				return;
			}

			model.currentNetwork = model.IsNew ? new network() : (network)Cloner.Clone(SelectedItem);

			if (!model.Validate())
			{
				return;
			}

			//Save changes.
			model.currentNetwork.storage_pool_id = Entity.Id;
			model.currentNetwork.name = (string)model.Name.Entity;
			model.currentNetwork.stp = model.IsStpEnabled;
			model.currentNetwork.description = (string)model.Description.Entity;
			model.currentNetwork.vlan_id = null;

			if (model.HasVLanTag)
			{
				model.currentNetwork.vlan_id = Convert.ToInt32(model.VLanTag.Entity.ToString());
			}

			model.newClusters = new List<VDSGroup>();
			
			foreach (SelectionTreeNodeModel selectionTreeNodeModel in model.ClusterTreeNodes)
			{
				if (selectionTreeNodeModel.IsSelectedNullable != null && selectionTreeNodeModel.IsSelectedNullable == true)
				{
					model.newClusters.Add((VDSGroup)selectionTreeNodeModel.Entity);
				}
			}
			List<VDSGroup> detachNetworkFromClusters = Linq.Except(model.OriginalClusters, model.newClusters);
			List<VdcActionParametersBase> actionParameters = new List<VdcActionParametersBase>();

			foreach (VDSGroup detachNetworkFromCluster in detachNetworkFromClusters)
			{
				actionParameters.Add(
					(VdcActionParametersBase)
					new AttachNetworkToVdsGroupParameter(detachNetworkFromCluster, model.currentNetwork));
			}

			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.DetachNetworkToVdsGroup, actionParameters,
				result =>
				{
					DataCenterNetworkModel networkModel = (DataCenterNetworkModel)result.State;
					network network = networkModel.currentNetwork;
					VdcReturnValueBase returnValue;
					if (networkModel.IsNew)
					{

						returnValue = Frontend.RunAction(VdcActionType.AddNetwork,
						new AddNetworkStoragePoolParameters(Entity.Id, network));

					}
					else
					{
						if ((bool)networkModel.IsEnabled.Entity)
						{
							returnValue = Frontend.RunAction(VdcActionType.UpdateNetwork,
								new AddNetworkStoragePoolParameters(Entity.Id, network));
						}
						else
						{
							returnValue = new VdcReturnValueBase() { Succeeded = true };
						}
					}

					if (returnValue != null && returnValue.Succeeded)
					{
						Guid networkId = networkModel.IsNew ? (guid)returnValue.ActionReturnValue : network.Id;
						List<VDSGroup> attachNetworkToClusters = Linq.Except(networkModel.newClusters, networkModel.OriginalClusters);
						List<VdcActionParametersBase>  actionParameters1 = new List<VdcActionParametersBase>();

						foreach (VDSGroup attachNetworkToCluster in attachNetworkToClusters)
						{
							actionParameters1.Add(
								(VdcActionParametersBase)
								new AttachNetworkToVdsGroupParameter(attachNetworkToCluster,
																	 new network() { Id = networkId, name = network.name }));
						}

						Frontend.RunMultipleAction(VdcActionType.AttachNetworkToVdsGroup, actionParameters1);
					}

					if (returnValue != null && returnValue.Succeeded)
					{
						Cancel();
					}
					networkModel.StopProgress();
				},
				model
			);
		}

		public void DetachClusters()
		{
			ConfirmationModel confirmModel = new ConfirmationModel();
			ConfirmWindow = confirmModel;
			confirmModel.Title = "Detach Network from ALL Clusters";
			confirmModel.HashName = "detach_network_from_all_clusters";
			confirmModel.Message =
				"You are about to detach the Network from all of the Clusters to which it is currentlyattached.\nAs a result, the Clusters' Hosts might become unreachable.\n\nAre you sure you want to continue?";
			confirmModel.Latch.IsAvailable = true;
			confirmModel.Commands.Add(
				new UICommand("OnDetachClusters", this)
				{
					Title = "OK",
					IsDefault = true
				});
			confirmModel.Commands.Add(
				new UICommand("CancelConfirmation", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void CancelConfirmation()
		{
			ConfirmWindow = null;
		}

		public void OnDetachClusters()
		{
			ConfirmationModel confirmationModel = (ConfirmationModel)ConfirmWindow;

			if (!confirmationModel.Validate())
			{
				return;
			}
			DataCenterNetworkModel model = (DataCenterNetworkModel)Window;
			network network = (network)SelectedItem;

			List<VdcActionParametersBase> actionParameters = new List<VdcActionParametersBase>();

			foreach (SelectionTreeNodeModel selectionTreeNodeModel in model.ClusterTreeNodes)
			{
				if (selectionTreeNodeModel.IsSelectedNullable != null && selectionTreeNodeModel.IsSelectedNullable == true)
				{
					selectionTreeNodeModel.IsSelectedNullable = false;
					actionParameters.Add((VdcActionParametersBase)new AttachNetworkToVdsGroupParameter((VDSGroup)selectionTreeNodeModel.Entity, network));
				}
			}

			List<VdcReturnValueBase> returnValueList = Frontend.RunMultipleAction(VdcActionType.DetachNetworkToVdsGroup, actionParameters);
			bool isSucceded = true;
			foreach (VdcReturnValueBase vdcReturnValueBase in returnValueList)
			{
				isSucceded &= vdcReturnValueBase.Succeeded;
			}
			
			CancelConfirmation();
			
			if (isSucceded)
			{
				model.OriginalClusters = new List<VDSGroup>(); 
				model.IsEnabled.Entity = true;
				model.DetachAllAvailable.Entity = !(bool)model.IsEnabled.Entity;
			}
			else
			{
				Cancel();
			}
		}

		public void Cancel()
		{
			Window = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		private void UpdateActionAvailability()
		{
			ArrayList selectedItems = (ArrayList)(SelectedItems ?? new ArrayList());

			bool anyRhevm = false;
			foreach (object item in selectedItems)
			{
				network network = (network)item;
				if (network.name == RHEVM_NETWORK)
				{
					anyRhevm = true;
					break;
				}
			}

			EditCommand.IsExecutionAllowed = selectedItems.Count == 1;
			RemoveCommand.IsExecutionAllowed = selectedItems.Count > 0 && !anyRhevm;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewCommand)
			{
				New();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}

			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if(command.Name == "DetachClusters")
			{
				DetachClusters();
			}
			else if(command.Name == "OnDetachClusters")
			{
				OnDetachClusters();
			}
			else if (command.Name == "CancelConfirmation")
			{
				CancelConfirmation();
			}
		}

		#region IFrontendMultipleQueryAsyncCallback Members

		public void Executed(FrontendMultipleQueryAsyncResult result)
		{
			network network = (network)SelectedItem;
			IList<VdcQueryReturnValue> returnValueList = result.ReturnValues;
			DataCenterNetworkModel model = (DataCenterNetworkModel)Window;
			List<network> clusterNetworkList = null;
			bool networkHasAttachedClusters = false;
			for (int i = 0; i < returnValueList.Count; i++)
			{
				VdcQueryReturnValue returnValue = returnValueList[i];
				if (returnValue.Succeeded && returnValue.ReturnValue != null)
				{
					clusterNetworkList = (List<network>)returnValue.ReturnValue;
					foreach (network clusterNetwork in clusterNetworkList)
					{
						if (clusterNetwork.Id.Equals(network.Id))
						{
							model.OriginalClusters.Add((VDSGroup)SelectionNodeList[i].Entity);
							SelectionNodeList[i].IsSelectedNullable = true;
							networkHasAttachedClusters = true;
							break;
						}
					}
				}
			}
			if (networkHasAttachedClusters)
			{
				model.IsEnabled.Entity = false;
				if (network.name != RHEVM_NETWORK)
				{
					model.DetachAllAvailable.Entity = !(bool)model.IsEnabled.Entity;
				}
			}

			model.ClusterTreeNodes = SelectionNodeList;
			if (network.name == RHEVM_NETWORK && SelectionNodeList.Count > 0)
			{
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				model.Commands.Add(
					new UICommand("OnSave", this)
						{
							Title = "OK",
							IsDefault = true
						});
				model.Commands.Add(
					new UICommand("Cancel", this)
						{
							Title = "Cancel",
							IsCancel = true
						});
			}
		}

		#endregion
	}
}
