using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.clusters;
using org.ovirt.engine.ui.uicommon.models.datacenters;
using org.ovirt.engine.ui.uicommon.models.storage;
using org.ovirt.engine.ui.uicommon.validation;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.BusinessEntities;
using VdcCommon.Interfaces;
using VdcFrontend;

namespace org.ovirt.engine.ui.uicommon.models.hosts
{
	public class ConfigureLocalStorageModel : Model
	{
		#region Properties

		public LocalStorageModel Storage { get; private set; }
		public DataCenterModel DataCenter { get; private set; }
		public ClusterModel Cluster { get; private set; }
		public EntityModel FormattedStorageName { get; private set; }

		private bool isEditedFlag;

		private bool editMode;
		public bool EditMode
		{
			get { return editMode; }
			set 
			{
				//once editing the view the flags stays true 
				if (isEditedFlag != true)
				{
					isEditedFlag = value;
				}
				editMode = value;
			}
		}

		private bool dontCreateDataCenter { get; set; }
		private bool dontCreateCluster { get; set; }
		private bool dontChangeHostCluster { get; set; }

		public bool DontCreateDataCenter 
		{
			get { return dontCreateDataCenter & !isEditedFlag; }
			set { dontCreateDataCenter = value; }
		}
		public bool DontCreateCluster
		{
			get { return dontCreateCluster & !isEditedFlag; }
			set { dontCreateCluster = value; }
		}
		public bool DontChangeHostCluster
		{
			get { return dontChangeHostCluster & !isEditedFlag; }
			set { dontChangeHostCluster = value; }
		}

		private string CommonName { get; set; }

		bool isGeneralTabValid;
		public bool IsGeneralTabValid
		{
			get { return isGeneralTabValid; }
			set
			{
				if (isGeneralTabValid != value)
				{
					isGeneralTabValid = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsGeneralTabValid"));
				}
			}
		}

		#endregion

		public ConfigureLocalStorageModel()
		{
			Storage = new LocalStorageModel();
			DataCenter = new DataCenterModel();
			DataCenter.Version.SelectedItemChangedEvent.addListener(this);
			Cluster = new ClusterModel();
			FormattedStorageName = new EntityModel();

			//Set the storage type to be Local.
			foreach (StorageType item in (List<StorageType>)DataCenter.StorageTypeList.Items)
			{
				if (item == StorageType.LOCALFS)
				{
					DataCenter.StorageTypeList.SelectedItem = item;
					break;
				}
			}

			IsGeneralTabValid = true;
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(ListModel.SelectedItemChangedEventDefinition) && sender == DataCenter.Version)
			{
				DataCenterVersion_SelectedItemChanged();
			}
		}

		private void DataCenterVersion_SelectedItemChanged()
		{
			Version version = (Version)DataCenter.Version.SelectedItem;

			//Keep in sync version for data center and cluster.
			Cluster.Version.SelectedItem = version;
		}

		internal bool Validate()
		{
			FormattedStorageName.ValidateEntity(new IValidation[] { new RegexValidation() { Expression = "^[A-Za-z0-9_-]+$", Message = "Name can contain only 'A-Z', 'a-z', '0-9', '_' or '-' characters." } });
			if (FormattedStorageName.Entity != null && !DataProvider.IsStorageDomainNameUnique((string)FormattedStorageName.Entity))
			{
				FormattedStorageName.IsValid = false;
				FormattedStorageName.InvalidityReasons.Add("Name must be unique.");
			}
			bool isStorageValid = Storage.Validate() && FormattedStorageName.IsValid;
			bool isDataCenterValid = true;
			if(!DontCreateDataCenter)
			{
				isDataCenterValid = DataCenter.Validate();
			}
			bool isClusterValid = true;
			if (!DontCreateCluster)
			{
				isClusterValid = Cluster.Validate(false);
			}

			IsGeneralTabValid =
				isStorageValid
				&& isDataCenterValid
				&& isClusterValid;

			return isStorageValid && isDataCenterValid && isClusterValid;
		}

		public void SetDefaultNames(VDS host, out string message)
		{
			message = null;
			CommonName = String.Format("{0}-Local", host.vds_name.Replace('.', '-'));
			storage_pool candidate = null;

			//selecet all possible DCs
			VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.Search,
				new SearchParameters(String.Format("DataCenter: name={0}", CommonName + "*"), SearchType.StoragePool));

			List<storage_pool> dataCenterList = new List<storage_pool>();
			List<VDSGroup> clusterList = null;
			if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
			{
				dataCenterList = Linq.Cast<storage_pool>((List<IVdcQueryable>)returnValue.ReturnValue);
			}
			//check if current settings suitable for local setup (in case just SD creation failed - re-using the same setup)
			bool useCurrentSettings = false;
			if (host.storage_pool_id != null)
			{
				storage_pool tempCandidate = DataProvider.GetDataCenterById(host.storage_pool_id);
				if (IsLocalDataCenterEmpty(tempCandidate))
				{
					candidate = tempCandidate;
					useCurrentSettings = true;
				}
				else
				{
					if (tempCandidate != null && tempCandidate.storage_pool_type == StorageType.LOCALFS)
					{
						message = "Note: Local Storage is already configured for this Host. The Host belongs to "
						          + host.storage_pool_name +
						          " with local Storage Domain. If OK is clicked - this Host will be moved to a new Data Center, and a new Local Storage Domain will be created. Hit Cancel to abort the operation.";
					}
				}
			}
			//check if there is other DC suitable for re-use
			if (candidate == null)
			{
				foreach (storage_pool dataCenter in dataCenterList)
				{
					//need to check if the new DC is without host.
					if (IsLocalDataCenterEmpty(dataCenter) &&  DataProvider.GetLocalStorageHost(dataCenter.name) == null)
					{
						candidate = dataCenter;
						break;
					}
				}
			}
			List<string> listNames = new List<string>();
			//in case we found a suitable candidte for re-use:
			if(candidate != null)
			{
				DataCenter.DataCenterId = candidate.Id;
				DataCenter.Name.Entity = candidate.name;
				DataCenter.Description.Entity = candidate.description;
				Version compVersion = candidate.compatibility_version;
				DataCenter.Version.SelectedItem = compVersion;
				Cluster.Version.SelectedItem = compVersion;
				DontCreateDataCenter = true;
				//if we use current settings there is no need to create cluster.
				if (useCurrentSettings)
				{
					Cluster.ClusterId = host.vds_group_id.Value;
					Cluster.Name.Entity = host.vds_group_name;
					VDSGroup cluster = DataProvider.GetClusterById(host.vds_group_id.Value);
					if(cluster != null)
					{
						Cluster.Description.Entity = cluster.description;
						Cluster.CPU.SelectedItem = new ServerCpu() {CpuName = cluster.cpu_name};
					}
					DontCreateCluster = true;
					DontChangeHostCluster = true;
				}
				//use differnt cluster
				else
				{
					//check the DC cluster list (for re-use)
					clusterList = DataProvider.GetClusterList(candidate.Id);
					//no clusters avilable - pick up new name.
					if(clusterList.Count == 0)
					{
						List<VDSGroup> listClusters = DataProvider.GetClusterList();
						listNames = new List<string>();
						foreach (VDSGroup cluster in listClusters)
						{
							listNames.Add(cluster.name);
						}
						Cluster.Name.Entity = AvailableName(listNames);
					}
					else
					{
						//use the DC cluster.
						Cluster.ClusterId = clusterList[0].ID;
						Cluster.Name.Entity = clusterList[0].name;
						Cluster.Description.Entity = clusterList[0].description;
						VDSGroup cluster = DataProvider.GetClusterById(Cluster.ClusterId.Value);
						if (cluster != null)
						{
							Cluster.CPU.SelectedItem = new ServerCpu() { CpuName = cluster.cpu_name };
						}
						DontCreateCluster = true;
						if (host.vds_group_id.Value.Equals(Cluster.ClusterId))
						{
							DontChangeHostCluster = true;
						}
					}
				}
			}
			else
			{
				//didn't found DC to re-use, so we select new names:
				listNames = new List<string>();
				foreach (storage_pool storagePool in dataCenterList)
				{
					listNames.Add(storagePool.name);
				}
				DataCenter.Name.Entity = AvailableName(listNames);
				
				//Choose a Data Center version corresponding to the host.
				if (!String.IsNullOrEmpty(host.supported_cluster_levels))
				{
					//the supported_cluster_levels are sorted.
					string[] array = host.supported_cluster_levels.Split(',');
					Version maxCombindVersion = null;

					for (int i = 0; i < array.Length; i++)
					{
						Version vdsVersion = new Version(array[i]);
						foreach (Version version in (IList<Version>)DataCenter.Version.Items)
						{
							if (version.Equals(vdsVersion) && version.CompareTo(maxCombindVersion) > 0)
							{
								maxCombindVersion = version;
							}
						}
					}
					if (maxCombindVersion != null)
					{
						DataCenter.Version.SelectedItem = maxCombindVersion;
						Cluster.Version.SelectedItem = maxCombindVersion;
					}
				}

				listNames = new List<string>();
				if (clusterList == null)
				{
					clusterList = DataProvider.GetClusterList();
				}

				foreach (VDSGroup cluster in clusterList)
				{
					listNames.Add(cluster.name);
				}
				Cluster.Name.Entity = AvailableName(listNames);
			}

			//Choose default CPU name to match host.
			if (host.CpuName != null && Cluster.CPU.SelectedItem != null)
			{
				Cluster.CPU.SelectedItem = Linq.FirstOrDefault((IList<ServerCpu>)Cluster.CPU.Items, new Linq.ServerCpuPredicate(host.CpuName.CpuName));
			}
			//always choose a avialable storage name.
			List<storage_domains> listStorageDomains = DataProvider.GetStorageDomainList();
			listNames = new List<string>();
			foreach (storage_domains storageDomain in listStorageDomains)
			{
				listNames.Add(storageDomain.storage_name);
			}
			FormattedStorageName.Entity = AvailableName(listNames);
		}

		private bool IsLocalDataCenterEmpty(storage_pool dataCenter)
		{
			if (dataCenter !=null
				&& dataCenter.storage_pool_type == StorageType.LOCALFS
				&& dataCenter.status == StoragePoolStatus.Uninitialized)
			{
				return true;
			}
			return false;
		}

		private string AvailableName(List<string> list)
		{
			string retVal = CommonName;
			List<int> notAvialbleNumberList = new List<int>();
			string temp;
			foreach (string str in list)
			{
				temp = str.Replace(CommonName, "");
				if (String.IsNullOrEmpty(temp))
				{
					temp = "0";
				}
				int tempInt;
				if (int.TryParse(temp, out tempInt))
				{
					notAvialbleNumberList.Add(tempInt);
				}
			}
			notAvialbleNumberList.Sort();
			int i = 0;
			for (i = 0; i < notAvialbleNumberList.Count; i++)
			{
				if (notAvialbleNumberList[i] == i)
				{
					continue;
				}
				break;
			}
			if (i > 0)
			{
				retVal = CommonName + i.ToString();
			}
			return retVal;
		}
	}
}
