using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using System.Timers;
using System.Windows.Threading;
using org.ovirt.engine.ui.uicommon.models.events;
using org.ovirt.engine.ui.uicommon.models.hosts;
using org.ovirt.engine.ui.uicommon.models.storage;
using org.ovirt.engine.ui.uicommon.models.vms;
using VdcCommon.Interfaces;

using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;
using System.Threading;
using Timer = System.Timers.Timer;

namespace org.ovirt.engine.ui.uicommon.models.monitor
{
	public class MonitorModel : SearchableListModel
	{
		#region Const

		// graph refresh interval in sec.
		private const int UPDATE_INTERVAL = 10;

		// default search string for VMs section of monitor model.
		private string VmDefaultSearchString;

		// search page size for VMs section:
		private const int VmSearchPageSize = 50;

		// default search string for Hosts section of monitor model.
		private const string HostDefaultSearchString = "Host: sortby cpu_usage desc";
		// search page size for Hosts section:
		private const int HostSearchPageSize = 30;

		// default search string for Storage section of monitor model.
		private const string StorageDefaultSearchString = "Storage:";
		// search page size for Storage section:
		private const int StorageSearchPageSize = 80;

		#endregion Const

		#region Commands

		public UICommand ServerNextPageCommand { get; private set; }
		public UICommand ServerPreviousPageCommand { get; private set; }
		public UICommand HostNextPageCommand { get; private set; }
		public UICommand HostPreviousPageCommand { get; private set; }
		public UICommand StorageNextPageCommand { get; private set; }
		public UICommand StoragePreviousPageCommand { get; private set; }

		#endregion Commands

		#region Properties

		private HostListModel HostListModel { get; set; }
		private VmListModel VmListModel { get; set; }
		private StorageListModel StorageListModel { get; set; }

		private ChartSeries hostCpuSeries;
		public ChartSeries HostCpuSeries
		{
			get { return hostCpuSeries; }
			private set
			{
				if (hostCpuSeries != value)
				{
					hostCpuSeries = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HostCpuSeries"));
				}
			}
		}

		private ChartSeries hostMemSeries;
		public ChartSeries HostMemSeries
		{
			get { return hostMemSeries; }
			private set
			{
				if (hostMemSeries != value)
				{
					hostMemSeries = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HostMemSeries"));
				}
			}
		}

		private ChartSeries hostNetSeries;
		public ChartSeries HostNetSeries
		{
			get { return hostNetSeries; }
			private set
			{
				if (hostNetSeries != value)
				{
					hostNetSeries = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HostNetSeries"));
				}
			}
		}

		private int hostsTotal;
		public int HostsTotal
		{
			get { return hostsTotal; }
			private set
			{
				if (hostsTotal != value)
				{
					hostsTotal = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HostsTotal"));
				}
			}
		}

		private int hostsActive;
		public int HostsActive
		{
			get { return hostsActive; }
			private set
			{
				if (hostsActive != value)
				{
					hostsActive = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HostsActive"));
				}
			}
		}

		private int hostsDown;
		public int HostsDown
		{
			get { return hostsDown; }
			private set
			{
				if (hostsDown != value)
				{
					hostsDown = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HostsDown"));
				}
			}
		}

		private int hostsMaintenance;
		public int HostsMaintenance
		{
			get { return hostsMaintenance; }
			private set
			{
				if (hostsMaintenance != value)
				{
					hostsMaintenance = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HostsMaintenance"));
				}
			}
		}

		private int hostsShowStart;
		public int HostsShowStart
		{
			get { return hostsShowStart; }
			private set
			{
				if (hostsShowStart != value)
				{
					hostsShowStart = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HostsShowStart"));
				}
			}
		}

		private int hostsShowEnd;
		public int HostsShowEnd
		{
			get { return hostsShowEnd; }
			private set
			{
				if (hostsShowEnd != value)
				{
					hostsShowEnd = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HostsShowEnd"));
				}
			}
		}

		private ChartSeries serverCpuSeries;
		public ChartSeries ServerCpuSeries
		{
			get { return serverCpuSeries; }
			private set
			{
				if (serverCpuSeries != value)
				{
					serverCpuSeries = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ServerCpuSeries"));
				}
			}
		}

		private ChartSeries serverMemSeries;
		public ChartSeries ServerMemSeries
		{
			get { return serverMemSeries; }
			private set
			{
				if (serverMemSeries != value)
				{
					serverMemSeries = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ServerMemSeries"));
				}
			}
		}

		private ChartSeries serverNetSeries;
		public ChartSeries ServerNetSeries
		{
			get { return serverNetSeries; }
			private set
			{
				if (serverNetSeries != value)
				{
					serverNetSeries = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ServerNetSeries"));
				}
			}
		}

		private int serversTotal;
		public int ServersTotal
		{
			get { return serversTotal; }
			private set
			{
				if (serversTotal != value)
				{
					serversTotal = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ServersTotal"));
				}
			}
		}

		private int serversDown;
		public int ServersDown
		{
			get { return serversDown; }
			private set
			{
				if (serversDown != value)
				{
					serversDown = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ServersDown"));
				}
			}
		}

		private int serversActive;
		public int ServersActive
		{
			get { return serversActive; }
			private set
			{
				if (serversActive != value)
				{
					serversActive = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ServersActive"));
				}
			}
		}

		private int serversActiveShowStart;
		public int ServersActiveShowStart
		{
			get { return serversActiveShowStart; }
			private set
			{
				if (serversActiveShowStart != value)
				{
					serversActiveShowStart = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ServersActiveShowStart"));
				}
			}
		}

		private int serversActiveShowEnd;
		public int ServersActiveShowEnd
		{
			get { return serversActiveShowEnd; }
			private set
			{
				if (serversActiveShowEnd != value)
				{
					serversActiveShowEnd = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ServersActiveShowEnd"));
				}
			}
		}

		private ChartSeries storageDomainSeries;
		public ChartSeries StorageDomainSeries
		{
			get { return storageDomainSeries; }
			private set
			{
				if (storageDomainSeries != value)
				{
					storageDomainSeries = value;
					OnPropertyChanged(new PropertyChangedEventArgs("StorageDomainSeries"));
				}
			}
		}

		private int storageDomainsTotal;
		public int StorageDomainsTotal
		{
			get { return storageDomainsTotal; }
			private set
			{
				if (storageDomainsTotal != value)
				{
					storageDomainsTotal = value;
					OnPropertyChanged(new PropertyChangedEventArgs("StorageDomainsTotal"));
				}
			}
		}

		private int storageDomainsActive;
		public int StorageDomainsActive
		{
			get { return storageDomainsActive; }
			private set
			{
				if (storageDomainsActive != value)
				{
					storageDomainsActive = value;
					OnPropertyChanged(new PropertyChangedEventArgs("StorageDomainsActive"));
				}
			}
		}

		private int storageDomainsShowStart;
		public int StorageDomainsShowStart
		{
			get { return storageDomainsShowStart; }
			private set
			{
				if (storageDomainsShowStart != value)
				{
					storageDomainsShowStart = value;
					OnPropertyChanged(new PropertyChangedEventArgs("StorageDomainsShowStart"));
				}
			}
		}

		private int storageDomainsShowEnd;
		public int StorageDomainsShowEnd
		{
			get { return storageDomainsShowEnd; }
			private set
			{
				if (storageDomainsShowEnd != value)
				{
					storageDomainsShowEnd = value;
					OnPropertyChanged(new PropertyChangedEventArgs("StorageDomainsShowEnd"));
				}
			}
		}

		public EventListModel EventList { get; private set; }

		#endregion

		private readonly Timer timer;
		private readonly SynchronizationContext syncContext;

		public MonitorModel()
		{
			Title = "Monitor";
			DefaultSearchString = "DataCenter:";
			SearchString = DefaultSearchString;

			InitVmDefaultSearchString();
			InitListModels();
			InitCommands();

			syncContext = SynchronizationContext.Current;
			timer = new Timer(200);
			timer.Elapsed += timer_Elapsed;
		}

		private void InitVmDefaultSearchString()
		{
			List<VMStatus> activeVmStatuses =
				new List<VMStatus>
				{
					VMStatus.Up,
					VMStatus.PoweringUp,
					VMStatus.PoweredDown,
					VMStatus.MigratingTo,
					VMStatus.WaitForLaunch,
					VMStatus.RebootInProgress,
					VMStatus.PoweringDown,
					VMStatus.Paused,
					VMStatus.Unknown
				};

			// arrVmStatusesSearchStrings will contains { "status=Up", "status=PoweringUp", ... }
			//			string[] arrVmStatusesSearchStrings = 
			//				activeVmStatuses.Select<VMStatus, string>(a => string.Format("status={0}", a)).ToArray();
			List<string> arrVmStatusesSearchStrings = new List<string>();
			foreach (VMStatus status in activeVmStatuses)
			{
				arrVmStatusesSearchStrings.Add(string.Format("status={0}", status));
			}


			// activeVmsSearchStringStatusesPart will contains "status=Up or status=PoweringUp or ..."
			//			string activeVmsSearchStringStatusesPart = string.Join(" or ", arrVmStatusesSearchStrings);
			string activeVmsSearchStringStatusesPart = string.Join(" or ", arrVmStatusesSearchStrings.ToArray());

			VmDefaultSearchString = string.Format("Vms: {0} sortby cpu_usage desc", activeVmsSearchStringStatusesPart);
		}

		private void InitListModels()
		{
			VmListModel =
				new VmListModel
				{
					DefaultSearchString = VmDefaultSearchString,
					SearchString = VmDefaultSearchString,
					SearchPageSize = VmSearchPageSize,
					IsAsync = false
				};

			HostListModel =
				new HostListModel
				{
					DefaultSearchString = HostDefaultSearchString,
					SearchString = HostDefaultSearchString,
					SearchPageSize = HostSearchPageSize,
					IsAsync = false
				};

			StorageListModel =
				new StorageListModel
				{
					DefaultSearchString = StorageDefaultSearchString,
					SearchString = StorageDefaultSearchString,
					SearchPageSize = StorageSearchPageSize,
					IsAsync = false
				};

			EventList =
				new EventListModel
				{
					SearchString = "Events: severity=error"
				};
		}

		private void InitCommands()
		{
			ServerNextPageCommand = new UICommand("SearchServerNextPage", this);
			ServerPreviousPageCommand = new UICommand("SearchServerPreviousPage", this);
			HostNextPageCommand = new UICommand("SearchHostNextPage", this);
			HostPreviousPageCommand = new UICommand("SearchHostPreviousPage", this);
			StorageNextPageCommand = new UICommand("SearchStorageNextPage", this);
			StoragePreviousPageCommand = new UICommand("SearchStoragePreviousPage", this);

			DisablePagingCommands();
		}

		private void DisablePagingCommands()
		{
			ServerNextPageCommand.IsExecutionAllowed = false;
			ServerPreviousPageCommand.IsExecutionAllowed = false;
			HostNextPageCommand.IsExecutionAllowed = false;
			HostPreviousPageCommand.IsExecutionAllowed = false;
			StorageNextPageCommand.IsExecutionAllowed = false;
			StoragePreviousPageCommand.IsExecutionAllowed = false;
		}

		private void RestorePagingCommands()
		{
			ServerNextPageCommand.IsExecutionAllowed = VmListModel.SearchNextPageCommand.IsExecutionAllowed;
			ServerPreviousPageCommand.IsExecutionAllowed = VmListModel.SearchPreviousPageCommand.IsExecutionAllowed;
			HostNextPageCommand.IsExecutionAllowed = HostListModel.SearchNextPageCommand.IsExecutionAllowed;
			HostPreviousPageCommand.IsExecutionAllowed = HostListModel.SearchPreviousPageCommand.IsExecutionAllowed;
			StorageNextPageCommand.IsExecutionAllowed = StorageListModel.SearchNextPageCommand.IsExecutionAllowed;
			StoragePreviousPageCommand.IsExecutionAllowed = StorageListModel.SearchPreviousPageCommand.IsExecutionAllowed;
		}

		private bool timerWasEnabled;

		private void SaveTimerStatusAndDisable()
		{
			if (timer.Enabled)
			{
				timerWasEnabled = true;
				timer.Enabled = false;
			}
		}

		private void RestoreTimerStatus()
		{
			timer.Enabled = timerWasEnabled;
		}

		public void SearchServerNextPage()
		{
			lock (timer)
			{
				SaveTimerStatusAndDisable();
			}
			DisablePagingCommands();

			VmListModel.SearchNextPageCommand.Execute();
			IEnumerable servers = VmListModel.Items ?? new List<IVdcQueryable>();
			Dictionary<string, int> statistics = DataProvider.GetSystemStatistics();
			syncContext.Send(UpdateServerChartData, new ArrayList { servers, statistics });

			RestorePagingCommands();
			RestoreTimerStatus();
		}

		public void SearchServerPreviousPage()
		{
			lock (timer)
			{
				SaveTimerStatusAndDisable();
			}
			DisablePagingCommands();

			VmListModel.SearchPreviousPageCommand.Execute();
			IEnumerable servers = VmListModel.Items ?? new List<IVdcQueryable>();
			Dictionary<string, int> statistics = DataProvider.GetSystemStatistics();
			syncContext.Send(UpdateServerChartData, new ArrayList { servers, statistics });

			RestorePagingCommands();
			RestoreTimerStatus();
		}

		public void SearchHostNextPage()
		{
			lock (timer)
			{
				SaveTimerStatusAndDisable();
			}
			DisablePagingCommands();

			HostListModel.SearchNextPageCommand.Execute();
			IEnumerable hosts = HostListModel.Items ?? new List<IVdcQueryable>();
			Dictionary<string, int> statistics = DataProvider.GetSystemStatistics();
			syncContext.Send(UpdateHostChartData, new ArrayList { hosts, statistics });

			RestorePagingCommands();
			RestoreTimerStatus();
		}

		public void SearchHostPreviousPage()
		{
			lock (timer)
			{
				SaveTimerStatusAndDisable();
			}
			DisablePagingCommands();

			HostListModel.SearchPreviousPageCommand.Execute();
			IEnumerable hosts = HostListModel.Items ?? new List<IVdcQueryable>();
			Dictionary<string, int> statistics = DataProvider.GetSystemStatistics();
			syncContext.Send(UpdateHostChartData, new ArrayList { hosts, statistics });

			RestorePagingCommands();
			RestoreTimerStatus();
		}

		public void SearchStorageNextPage()
		{
			lock (timer)
			{
				SaveTimerStatusAndDisable();
			}
			DisablePagingCommands();

			StorageListModel.SearchNextPageCommand.Execute();
			IEnumerable storageDomains = StorageListModel.Items ?? new List<IVdcQueryable>();
			Dictionary<string, int> statistics = DataProvider.GetSystemStatistics();
			syncContext.Send(UpdateStorageChartData, new ArrayList { storageDomains, statistics });

			RestorePagingCommands();
			RestoreTimerStatus();
		}

		public void SearchStoragePreviousPage()
		{
			lock (timer)
			{
				SaveTimerStatusAndDisable();
			}
			DisablePagingCommands();

			StorageListModel.SearchPreviousPageCommand.Execute();
			IEnumerable storageDomains = StorageListModel.Items ?? new List<IVdcQueryable>();
			Dictionary<string, int> statistics = DataProvider.GetSystemStatistics();
			syncContext.Send(UpdateStorageChartData, new ArrayList { storageDomains, statistics });

			RestorePagingCommands();
			RestoreTimerStatus();
		}

		public override bool IsSearchStringMatch(string searchString)
		{
			return searchString.Trim().ToLower().StartsWith("datacenter");
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			timer.Interval = 200;
			timer.Start();

			EventList.SearchCommand.Execute();
		}

		internal override void EnsureAsyncSearchStopped()
		{
			base.EnsureAsyncSearchStopped();

			timer.Stop();
			HostCpuSeries = null;
			HostMemSeries = null;
			HostNetSeries = null;

			Logger.Debug("Monitor: EnsureAsyncSearchStopped");
			EventList.EnsureAsyncSearchStopped();
		}

		public void BuildHostSeries()
		{
			Dictionary<string, int> statistics = DataProvider.GetSystemStatistics();
			BuildHostSeries(statistics);
		}

		private void BuildHostSeries(Dictionary<string, int> statistics)
		{
			HostListModel.SearchCommand.Execute();
			IEnumerable hosts = HostListModel.Items ?? new List<IVdcQueryable>();
			syncContext.Send(UpdateHostChartData, new ArrayList { hosts, statistics });
		}

		private void UpdateHostChartData(object obj)
		{
			ArrayList array = (ArrayList)obj;

			// statistics:
			Dictionary<string, int> statistics = (Dictionary<string, int>)array[1];
			HostsTotal = statistics.ContainsKey("total_vds") ? statistics["total_vds"] : 0;
			HostsActive = statistics.ContainsKey("active_vds") ? statistics["active_vds"] : 0;
			HostsDown = statistics.ContainsKey("down_vds") ? statistics["down_vds"] : 0;
			HostsMaintenance = statistics.ContainsKey("maintenance_vds") ? statistics["maintenance_vds"] : 0;

			// chart data:
			IEnumerable hosts = (IEnumerable)array[0];

			ChartSeries cpuSeries = new ChartSeries();
			ChartSeries memSeries = new ChartSeries();
			ChartSeries netSeries = new ChartSeries();

			foreach (object item in hosts)
			{
				VDS host = (VDS)item;
				cpuSeries.Items.Add(
					new ChartSeriesModel
					{
						X = string.Format("{0}\nCPU: {1}%", host.vds_name, host.usage_cpu_percent.GetValueOrDefault()),
						Y = host.usage_cpu_percent.GetValueOrDefault(),
						Status = host.status
					});

				memSeries.Items.Add(
					new ChartSeriesModel
					{
						X = string.Format("{0}\nMemory: {1}%", host.vds_name, host.usage_mem_percent.GetValueOrDefault()),
						Y = host.usage_mem_percent.GetValueOrDefault(),
						Status = host.status
					});

				netSeries.Items.Add(
					new ChartSeriesModel
					{
						X = string.Format("{0}\nNetwork: {1}%", host.vds_name, host.usage_network_percent.GetValueOrDefault()),
						Y = host.usage_network_percent.GetValueOrDefault(),
						Status = host.status
					});
			}

			HostCpuSeries = cpuSeries;
			HostMemSeries = memSeries;
			HostNetSeries = netSeries;

			HostsShowStart = ((HostListModel.SearchPageNumber - 1) * HostListModel.SearchPageSize) + 1;
			HostsShowEnd = HostsShowStart - 1 + cpuSeries.Items.Count;
			// we don't use the HostListModel's Items since it is 
			// IEnumerable and doesn't have the Count property or method.
		}

		public void BuildServerSeries()
		{
			Dictionary<string, int> statistics = DataProvider.GetSystemStatistics();
			BuildServerSeries(statistics);
		}

		private void BuildServerSeries(Dictionary<string, int> statistics)
		{
			VmListModel.SearchCommand.Execute();
			IEnumerable servers = VmListModel.Items ?? new List<IVdcQueryable>();
			syncContext.Send(UpdateServerChartData, new ArrayList { servers, statistics });
		}

		private void UpdateServerChartData(object obj)
		{
			ArrayList array = (ArrayList)obj;

			// statistics:
			Dictionary<string, int> statistics = (Dictionary<string, int>)array[1];
			ServersTotal = statistics.ContainsKey("total_vms") ?
				statistics["total_vms"] :
				0;

			ServersActive = statistics.ContainsKey("active_vms") ?
				statistics["active_vms"] :
				0;

			ServersDown = statistics.ContainsKey("down_vms") ?
				statistics["down_vms"] :
				0;

			// chart data:
			IEnumerable servers = (IEnumerable)array[0];

			ChartSeries cpuSeries = new ChartSeries();
			ChartSeries memSeries = new ChartSeries();
			ChartSeries netSeries = new ChartSeries();

			foreach (VM server in servers)
			{
				cpuSeries.Items.Add(
					new ChartSeriesModel
					{
						X = string.Format("{0}\nCPU: {1}%", server.vm_name, server.usage_cpu_percent.GetValueOrDefault()),
						Y = server.usage_cpu_percent.GetValueOrDefault(),
						Status = server.status
					});

				memSeries.Items.Add(
					new ChartSeriesModel
					{
						X = string.Format("{0}\nMemory: {1}%", server.vm_name, server.usage_mem_percent.GetValueOrDefault()),
						Y = server.usage_mem_percent.GetValueOrDefault(),
						Status = server.status
					});

				netSeries.Items.Add(
					new ChartSeriesModel
					{
						X = string.Format("{0}\nNetwork: {1}%", server.vm_name, server.usage_network_percent.GetValueOrDefault()),
						Y = server.usage_network_percent.GetValueOrDefault(),
						Status = server.status
					});
			}

			ServerCpuSeries = cpuSeries;
			ServerMemSeries = memSeries;
			ServerNetSeries = netSeries;

			ServersActiveShowStart = ((VmListModel.SearchPageNumber - 1) * VmListModel.SearchPageSize) + 1;
			ServersActiveShowEnd = ServersActiveShowStart - 1 + cpuSeries.Items.Count;
			// we don't use the VmListModel's Items since it is 
			// IEnumerable and doesn't have the Count property or method.
		}

		public void BuildStorageSeries()
		{
			Dictionary<string, int> statistics = DataProvider.GetSystemStatistics();
			BuildStorageSeries(statistics);
		}

		private void BuildStorageSeries(Dictionary<string, int> statistics)
		{
			StorageListModel.SearchCommand.Execute();
			IEnumerable storageDomains = StorageListModel.Items ?? new List<IVdcQueryable>();
			syncContext.Send(UpdateStorageChartData, new ArrayList { storageDomains, statistics });
		}

		private void UpdateStorageChartData(object obj)
		{
			ArrayList array = (ArrayList)obj;

			// statistics:
			Dictionary<string, int> statistics = (Dictionary<string, int>)array[1];

			StorageDomainsTotal =
				statistics.ContainsKey("total_storage_domains") ?
				statistics["total_storage_domains"] :
				0;

			StorageDomainsActive =
				statistics.ContainsKey("active_storage_domains") ?
				statistics["active_storage_domains"] :
				0;

			// chart data:
			IEnumerable storageDomains = (IEnumerable)array[0];

			ChartSeries storageSeries = new ChartSeries();

			foreach (object item in storageDomains)
			{
				storage_domains storageDomain = (storage_domains)item;
				int available = storageDomain.available_disk_size.GetValueOrDefault();
				int used = storageDomain.used_disk_size.GetValueOrDefault();
				int total = available + used;

				storageSeries.Items.Add(
					new ChartSeriesModel
					{
						X = String.Format("{0}\nAvail.: {1} GB\nUsed: {2} GB\nTotal: {3} GB",
								storageDomain.storage_name,
								available,
								used,
								total),
						Y = total > 0 ? 100 * used / total : 0,
						Status = storageDomain.storage_domain_shared_status
					});
			}

			storageSeries.SortByY();

			StorageDomainSeries = storageSeries;

			StorageDomainsShowStart = ((StorageListModel.SearchPageNumber - 1) * StorageListModel.SearchPageSize) + 1;
			StorageDomainsShowEnd = StorageDomainsShowStart - 1 + storageSeries.Items.Count;
			// we don't use the StorageListModel's Items since it is 
			// IEnumerable and doesn't have the Count property or method.
		}

		private void timer_Elapsed(object sender, ElapsedEventArgs e)
		{
			lock (timer)
			{
				timer.Interval = (UPDATE_INTERVAL * 1000);
				if (timer.Enabled)
				// if timer is not enabled, someone has disabled it from outside
				// this method, so the same someoneshould enable it when it ends.
				{
					timer.Enabled = false;

					DisablePagingCommands();
					Dictionary<string, int> statistics = DataProvider.GetSystemStatistics();
					BuildHostSeries(statistics);
					BuildServerSeries(statistics);
					BuildStorageSeries(statistics);
					RestorePagingCommands();

					timer.Enabled = true;
				}
			}
		}
	}
}
