using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.ComponentModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.validation;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.VdcQueries;
using VdcCommon.BusinessEntities;
using VdcFrontend;

namespace org.ovirt.engine.ui.uicommon.models.storage
{
	public abstract class SanStorageModelBase : ListModel, IStorageModel
	{
		#region Commands

		public UICommand UpdateCommand { get; private set; }
		public UICommand LoginAllCommand { get; private set; }
		public UICommand DiscoverTargetsCommand { get; private set; }

		#endregion

		#region Properties

		public StorageModel Container { get; set; }
		public StorageDomainType Role { get; set; }
		public abstract StorageType Type { get; }

		public EntityModel Address { get; private set; }
		public EntityModel Port { get; private set; }
		public EntityModel UserName { get; private set; }
		public EntityModel Password { get; private set; }
		public EntityModel UseUserAuth { get; private set; }

		private bool proposeDiscoverTargets;
		public bool ProposeDiscoverTargets
		{
			get { return proposeDiscoverTargets; }
			set
			{
				if (proposeDiscoverTargets != value)
				{
					proposeDiscoverTargets = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ProposeDiscoverTargets"));
				}
			}
		}

		#endregion

		private bool loginAllInProgress;

		protected SanStorageModelBase()
		{
			UpdateCommand = new UICommand("Update", this);
			LoginAllCommand = new UICommand("LoginAll", this) { IsExecutionAllowed = false };
			DiscoverTargetsCommand = new UICommand("DiscoverTargets", this);

			Address = new EntityModel();
			Port = new EntityModel { Entity = "3260" };
			UserName = new EntityModel();
			Password = new EntityModel();
			UseUserAuth = new EntityModel { Entity = false };
			UseUserAuth.EntityChangedEvent.addListener(this);

			UpdateUserAuthFields();
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(SanTargetModel.LoggedInEventDefinition))
			{
				SanTargetModel_LoggedIn(sender, args);
			}
			else if (ev.Equals(EntityChangedEventDefinition))
			{
				UseUserAuth_EntityChanged(sender, args);
			}
		}

		private void SanTargetModel_LoggedIn(object sender, EventArgs args)
		{
			VDS host = (VDS)Container.Host.SelectedItem;
			if (host == null)
			{
				return;
			}

			SanTargetModel model = (SanTargetModel)sender;

			storage_server_connections connection =
				new storage_server_connections
				{
					portal = "0",
					storage_type = StorageType.ISCSI,
					user_name = (bool)UseUserAuth.Entity ? (string)UserName.Entity : String.Empty,
					password = (bool)UseUserAuth.Entity ? (string)Password.Entity : String.Empty,
					iqn = model.Name,
					connection = model.Address,
					port = Convert.ToString(model.Port)
				};

			VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.ConnectStorageToVds,
				new StorageServerConnectionParametersBase(connection, host.vds_id));

			if (returnValue != null && returnValue.Succeeded)
			{
				model.IsLoggedIn = true;
				model.LoginCommand.IsExecutionAllowed = false;

				if (!loginAllInProgress)
				{
					UpdateInternal();
				}
			}
		}

		private void LoginAll()
		{
			//Cast to list of SanTargetModel because we get call 
			//to this method only from target/LUNs mode.

			loginAllInProgress = true;
			bool updateRequired = false;
			IList<SanTargetModel> items = (IList<SanTargetModel>)Items;

			foreach (SanTargetModel item in items)
			{
				if (!item.IsLoggedIn)
				{
					item.LoginCommand.Execute();
					updateRequired = true;
				}
			}

			if (updateRequired)
			{
				UpdateInternal();
			}

			loginAllInProgress = false;
		}

		private void DiscoverTargets()
		{
			if (Container.Progress != null)
			{
				return;
			}

			if (!ValidateDiscoverTargetFields())
			{
				return;
			}

			VDS host = (VDS)Container.Host.SelectedItem;

			DiscoverSendTargetsQueryParameters parameters =
				new DiscoverSendTargetsQueryParameters(host.vds_id,
					new storage_server_connections
					{
						connection = ((string)Address.Entity).Trim(),
						port = ((string)Port.Entity).Trim(),
						portal = "0",
						storage_type = StorageType.ISCSI,
						user_name = (bool)UseUserAuth.Entity ? (string)UserName.Entity : String.Empty,
						password = (bool)UseUserAuth.Entity ? (string)Password.Entity : String.Empty
					});

			Message = null;
			Container.StartProgress(null);

			Frontend.RunQuery(VdcQueryType.DiscoverSendTargets, parameters, new AsyncQuery(this,
				(target, returnValue) =>
				{
					SanStorageModelBase model = (SanStorageModelBase)target;

					object result = ((VdcQueryReturnValue)returnValue).ReturnValue;

					model.PostDiscoverTargetsInternal(result != null
						? (List<storage_server_connections>)(valueObjectEnumerableList)result
						: new List<storage_server_connections>());
				},
				true)
			);
		}

		private void PostDiscoverTargetsInternal(List<storage_server_connections> items)
		{
			List<SanTargetModel> newItems = new List<SanTargetModel>();

			foreach (storage_server_connections a in items)
			{
				SanTargetModel model =
					new SanTargetModel
					{
						Address = a.connection,
						Port = a.port,
						Name = a.iqn,
						Luns = new ObservableCollection<LunModel>()
					};
				model.LoggedInEvent.addListener(this);

				newItems.Add(model);
			}

			Container.StopProgress();

			if (items.Count == 0)
			{
				Message = "No new devices were found. This may be due to either: incorrect multipath configuration on the Host or wrong address of the iscsi target or a failure to authenticate on the target device. Please consult your Storage Administrator.";
			}

			PostDiscoverTargets(newItems);
		}

		protected virtual void PostDiscoverTargets(List<SanTargetModel> newItems)
		{
		}

		private bool ValidateDiscoverTargetFields()
		{
			Container.Host.ValidateSelectedItem(new[] { new NotEmptyValidation() });

			Address.ValidateEntity(new IValidation[] { new NotEmptyValidation() });

			Port.ValidateEntity(
				new IValidation[]
				{
					new NotEmptyValidation(),
					new IntegerValidation {Minimum = 0, Maximum = 65535}
				});

			if ((bool)UseUserAuth.Entity)
			{
				UserName.ValidateEntity(new IValidation[] { new NotEmptyValidation() });
				Password.ValidateEntity(new IValidation[] { new NotEmptyValidation() });
			}

			return Container.Host.IsValid
				   && Address.IsValid
				   && Port.IsValid
				   && UserName.IsValid
				   && Password.IsValid;
		}

		public virtual bool Validate()
		{
			return true;
		}

		private void UseUserAuth_EntityChanged(object sender, EventArgs args)
		{
			UpdateUserAuthFields();
		}

		private void UpdateUserAuthFields()
		{
			UserName.IsValid = true;
			UserName.IsChangable = (bool)UseUserAuth.Entity;

			Password.IsValid = true;
			Password.IsChangable = (bool)UseUserAuth.Entity;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == UpdateCommand)
			{
				Update();
			}
			else if (command == LoginAllCommand)
			{
				LoginAll();
			}
			else if (command == DiscoverTargetsCommand)
			{
				DiscoverTargets();
			}
		}

		protected virtual void Update()
		{
			UpdateInternal();
			IsValid = true;
		}

		protected virtual void UpdateInternal()
		{
		}

		protected void UpdateLoginAllAvailability()
		{
			IList<SanTargetModel> items = (IList<SanTargetModel>)Items;

			//Allow login all command when there at least one target that may be logged in.
			bool allow = false;

			foreach (SanTargetModel item in items)
			{
				if (!item.IsLoggedIn)
				{
					allow = true;
					break;
				}
			}

			LoginAllCommand.IsExecutionAllowed = allow;
		}
	}


	public abstract class SanStorageModel : SanStorageModelBase
	{
		#region Properties

		private bool isGrouppedByTarget;
		/// <summary>
		/// Gets or sets the value determining whether the
		/// items containing target/LUNs or LUN/targets.
		/// </summary>
		public bool IsGrouppedByTarget
		{
			get { return isGrouppedByTarget; }
			set
			{
				if (isGrouppedByTarget != value)
				{
					isGrouppedByTarget = value;
					IsGrouppedByTargetChanged();
					OnPropertyChanged(new PropertyChangedEventArgs("IsGrouppedByTarget"));
				}
			}
		}

		private bool isAllLunsSelected;
		public bool IsAllLunsSelected
		{
			get { return isAllLunsSelected; }
			set
			{
				if (isAllLunsSelected != value)
				{
					isAllLunsSelected = value;
					IsAllLunsSelectedChanged();
					OnPropertyChanged(new PropertyChangedEventArgs("IsAllLunsSelected"));
				}
			}
		}

		private string getLUNsFailure;
		public string GetLUNsFailure
		{
			get { return getLUNsFailure; }
			set
			{
				if (getLUNsFailure != value)
				{
					getLUNsFailure = value;
					OnPropertyChanged(new PropertyChangedEventArgs("GetLUNsFailure"));
				}
			}
		}

		#endregion

		private readonly IList<LunModel> includedLUNs;
		private readonly List<SanTargetModel> lastDiscoveredTargets;

		protected SanStorageModel()
		{
			includedLUNs = new List<LunModel>();
			lastDiscoveredTargets = new List<SanTargetModel>();

			InitializeItems(null, null);
		}

		protected override void PostDiscoverTargets(List<SanTargetModel> newItems)
		{
			base.PostDiscoverTargets(newItems);

			InitializeItems(null, newItems);

			//Remember all discovered targets.
			lastDiscoveredTargets.Clear();
			lastDiscoveredTargets.AddRange(newItems);
		}

		protected override void Update()
		{
			lastDiscoveredTargets.Clear();

			base.Update();
		}

		protected override void UpdateInternal()
		{
			base.UpdateInternal();

			if (Container.Progress != null)
			{
				return;
			}

			VDS host = (VDS)Container.Host.SelectedItem;
			if (host == null)
			{
				ProposeDiscover();
				return;
			}


			ClearItems();
			InitializeItems(null, null);

			Container.StartProgress(null);

			Frontend.RunQuery(VdcQueryType.GetDeviceList, new GetDeviceListQueryParameters(host.vds_id, Type), new AsyncQuery(this,
				(target, returnValue) =>
				{
					SanStorageModel model = (SanStorageModel)target;

					VdcQueryReturnValue response = (VdcQueryReturnValue)returnValue;
					if (response.Succeeded)
					{
						model.ApplyData((List<LUNs>)response.ReturnValue, false);
					}
					else
					{
						model.GetLUNsFailure = "Could not retrieve LUNs, please check your storage.";
					}

					model.Container.StopProgress();
				},
				true)
			);
		}

		private void ClearItems()
		{
			if (Items == null)
			{
				return;
			}

			if (IsGrouppedByTarget)
			{
				IList<SanTargetModel> items = (IList<SanTargetModel>)Items;

				foreach (SanTargetModel target in Linq.ToList(items))
				{
					bool found = false;

					//Ensure remove targets that are not in last dicovered targets list.
					if (Linq.FirstOrDefault(lastDiscoveredTargets, new Linq.TargetPredicate(target)) != null)
					{
						found = true;
					}
					else
					{
						//Ensure remove targets that are not contain already included LUNs.
						foreach (LunModel lun in target.Luns)
						{
							LunModel foundItem = Linq.FirstOrDefault(includedLUNs, new Linq.LunPredicate(lun));
							if (foundItem == null)
							{
								found = true;
								break;
							}
						}
					}

					if (!found)
					{
						items.Remove(target);
					}
				}
			}
			else
			{
				IList<LunModel> items = (IList<LunModel>)Items;

				//Ensure remove targets that are not contain already included LUNs.
				foreach (LunModel lun in Linq.ToList(items))
				{
					LunModel foundItem = Linq.FirstOrDefault(includedLUNs, new Linq.LunPredicate(lun));
					if (foundItem == null)
					{
						items.Remove(lun);
					}
				}
			}
		}

		/// <summary>
		/// Creates model items from the provided list of business entities.
		/// </summary>
		public void ApplyData(IList<LUNs> source, bool isIncluded)
		{
			List<LunModel> newItems = new List<LunModel>();

			foreach (LUNs a in source)
			{
				if (a.LunType == Type || a.LunType == StorageType.UNKNOWN)
				{
					List<SanTargetModel> targets = new List<SanTargetModel>();
					foreach (storage_server_connections b in a.LunConnections)
					{
						SanTargetModel model =
							new SanTargetModel
							{
								Address = b.connection,
								Port = b.port,
								Name = b.iqn,
								IsSelected = true,
								IsLoggedIn = true,
								Luns = new ObservableCollection<LunModel>()
							};
						model.LoginCommand.IsExecutionAllowed = false;

						targets.Add(model);
					}

					var lun =
						new LunModel
						{
							LunId = a.LUN_id,
							VendorId = a.VendorId,
							ProductId = a.ProductId,
							Serial = a.Serial,
							Multipathing = a.PathCount,
							Targets = targets,
							Size = a.DeviceSize,
							IsAccessible = a.Accessible,
							IsIncluded = isIncluded,
							IsSelected = isIncluded
						};
					newItems.Add(lun);

					//Remember included LUNs to prevent their removal while updating items.
					if (isIncluded)
					{
						includedLUNs.Add(lun);
					}
				}
			}

			InitializeItems(newItems, null);
			ProposeDiscover();
		}

		private void IsGrouppedByTargetChanged()
		{
			InitializeItems(null, null);
		}

		/// <summary>
		/// Organizes items according to the current groupping flag.
		/// When new items provided takes them in account and add to the Items collection.
		/// </summary>
		private void InitializeItems(IList<LunModel> newLuns, IList<SanTargetModel> newTargets)
		{
			if (IsGrouppedByTarget)
			{
				if (Items == null)
				{
					Items = new ObservableCollection<SanTargetModel>();
				}
				else
				{
					//Convert to list of another type as neccessary.
					if (TypeUtil.IsListOf<LunModel>(Items))
					{
						Items = ToTargetModelList((IList<LunModel>)Items);
					}
				}

				IList<SanTargetModel> items = (IList<SanTargetModel>)Items;

				//Add new targets.
				if (newTargets != null)
				{
					foreach (SanTargetModel newItem in newTargets)
					{
						if (Linq.FirstOrDefault(items, new Linq.TargetPredicate(newItem)) == null)
						{
							items.Add(newItem);
						}
					}
				}

				//Merge luns into targets.
				if (newLuns != null)
				{
					MergeLunsToTargets(newLuns, items);
				}

				UpdateLoginAllAvailability();
			}
			else
			{
				if (Items == null)
				{
					Items = new ObservableCollection<LunModel>();
				}
				else
				{
					//Convert to list of another type as neccessary.
					if (TypeUtil.IsListOf<SanTargetModel>(Items))
					{
						Items = ToLunModelList((IList<SanTargetModel>)Items);
					}
				}

				IList<LunModel> items = (IList<LunModel>)Items;

				//Add new LUNs.
				if (newLuns != null)
				{
					foreach (LunModel newItem in newLuns)
					{
						if (Linq.FirstOrDefault(items, new Linq.LunPredicate(newItem)) == null)
						{
							items.Add(newItem);
						}
					}
				}
			}
		}

		private void MergeLunsToTargets(IList<LunModel> newLuns, IList<SanTargetModel> targets)
		{
			foreach (LunModel lun in newLuns)
			{
				foreach (SanTargetModel target in lun.Targets)
				{
					SanTargetModel item = Linq.FirstOrDefault(targets, new Linq.TargetPredicate(target));
					if (item == null)
					{
						item = target;
						targets.Add(item);
					}

					if (Linq.FirstOrDefault(item.Luns, new Linq.LunPredicate(lun)) == null)
					{
						item.Luns.Add(lun);
					}
				}
			}
		}

		private IList<SanTargetModel> ToTargetModelList(IList<LunModel> source)
		{
			ObservableCollection<SanTargetModel> list = new ObservableCollection<SanTargetModel>();

			foreach (LunModel lun in source)
			{
				foreach (SanTargetModel target in lun.Targets)
				{
					SanTargetModel item = Linq.FirstOrDefault(list, new Linq.TargetPredicate(target));
					if (item == null)
					{
						item = target;
						list.Add(item);
					}

					if (Linq.FirstOrDefault(item.Luns, new Linq.LunPredicate(lun)) == null)
					{
						item.Luns.Add(lun);
					}
				}
			}

			//Merge with last discovered targets list.
			foreach (SanTargetModel target in lastDiscoveredTargets)
			{
				if (Linq.FirstOrDefault(list, new Linq.TargetPredicate(target)) == null)
				{
					list.Add(target);
				}
			}

			return list;
		}

		private IList<LunModel> ToLunModelList(IList<SanTargetModel> source)
		{
			ObservableCollection<LunModel> list = new ObservableCollection<LunModel>();

			foreach (SanTargetModel target in source)
			{
				foreach (LunModel lun in target.Luns)
				{
					LunModel item = Linq.FirstOrDefault(list, new Linq.LunPredicate(lun));
					if (item == null)
					{
						item = lun;
						list.Add(item);
					}

					if (Linq.FirstOrDefault(item.Targets, new Linq.TargetPredicate(target)) == null)
					{
						item.Targets.Add(target);
					}
				}
			}

			return list;
		}

		private void ProposeDiscover()
		{
			if (!ProposeDiscoverTargets && (Items == null || Linq.Count(Items) == 0))
			{
				ProposeDiscoverTargets = true;
			}
		}

		private void IsAllLunsSelectedChanged()
		{
			if (!IsGrouppedByTarget)
			{
				IList<LunModel> items = (IList<LunModel>)Items;
				foreach (LunModel lun in items)
				{
					if (!lun.IsIncluded && lun.IsAccessible)
					{
						lun.IsSelected = IsAllLunsSelected;
					}
				}
			}
		}

		public List<LunModel> AddedLuns
		{
			get
			{
				List<LunModel> luns = new List<LunModel>();
				if (IsGrouppedByTarget)
				{
					IList<SanTargetModel> items = (IList<SanTargetModel>)Items;
					foreach (SanTargetModel item in items)
					{
						foreach (LunModel lun in item.Luns)
						{
							if (lun.IsSelected && !lun.IsIncluded && Linq.FirstOrDefault(luns, new Linq.LunPredicate(lun)) == null)
							{
								luns.Add(lun);
							}
						}
					}
				}
				else
				{
					IList<LunModel> items = (IList<LunModel>)Items;
					foreach (LunModel lun in items)
					{
						if (lun.IsSelected && !lun.IsIncluded && Linq.FirstOrDefault(luns, new Linq.LunPredicate(lun)) == null)
						{
							luns.Add(lun);
						}
					}
				}

				return luns;
			}
		}

		public override bool Validate()
		{
			IsValid = AddedLuns.Count > 0 || includedLUNs.Count > 0;

			return base.Validate() && IsValid;
		}
	}


	public class IscsiStorageModel : SanStorageModel
	{
		public override StorageType Type
		{
			get { return StorageType.ISCSI; }
		}
	}


	public class FcpStorageModel : SanStorageModel
	{
		public override StorageType Type
		{
			get { return StorageType.FCP; }
		}
	}
}