using System;
using System.Collections.Generic;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.templates;
using VdcCommon.VdcQueries;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using System.Windows.Threading;
using System.Threading;
using org.ovirt.engine.ui.uicompat;
using System.ComponentModel;

namespace org.ovirt.engine.ui.uicommon.models.storage
{
	public class TemplateBackupModel : ManageBackupModel, ITaskTarget
	{
		#region Properties

		public new Nullable<KeyValuePair<VmTemplate, List<DiskImage>>> SelectedItem
		{
			get { return (Nullable<KeyValuePair<VmTemplate, List<DiskImage>>>)base.SelectedItem; }
			set { base.SelectedItem = value; }
		}

		#endregion

		public TemplateBackupModel()
		{
			Title = "Template Import";
		}

		protected override void Remove()
		{
			base.Remove();

			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Backed up Template(s)";
			model.HashName = "remove_backed_up_template";
			model.Message = "Template(s)";
			//model.Items = SelectedItems.Cast<KeyValuePair<VmTemplate, List<DiskImage>>>().Select(a => a.Key.name);

			List<string> items = new List<string>();
			foreach (object a in SelectedItems)
			{
				KeyValuePair<VmTemplate, List<DiskImage>> item = (KeyValuePair<VmTemplate, List<DiskImage>>)a;
				VmTemplate template = item.Key;
				items.Add(template.name);
			}
			model.Items = items;

			model.Note = "Note: The deleted items might still appear on the sub-tab, since the remove operation might be long. Use the Refresh button, to get the updated status.";

			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnRemove()
		{
			storage_pool pool = DataProvider.GetFirstStoragePoolByStorageDomain(Entity.id);
			//Frontend.RunMultipleActions(VdcActionType.RemoveVmTemplateFromImportExport,
			//	SelectedItems.Cast<KeyValuePair<VmTemplate, List<DiskImage>>>()
			//	.Select(a => (VdcActionParametersBase)new VmTemplateImportExportParameters(a.Key.vmt_guid, Entity.id, pool.id))
			//	.ToList()
			//);
			List<VdcActionParametersBase> prms = new List<VdcActionParametersBase>();
			foreach (object a in SelectedItems)
			{
				KeyValuePair<VmTemplate, List<DiskImage>> item = (KeyValuePair<VmTemplate, List<DiskImage>>)a;
				VmTemplate template = item.Key;
				prms.Add(new VmTemplateImportExportParameters(template.Id, Entity.id, pool.Id));
			}

			Frontend.RunMultipleAction(VdcActionType.RemoveVmTemplateFromImportExport, prms);


			Cancel();
			OnEntityChanged();
		}

		protected override void Restore()
		{
			base.Restore();

			if (Window != null)
			{
				return;
			}

			ImportTemplateModel model = new ImportTemplateModel();
			Window = model;
			model.Title = "Import Template(s)";
			model.HashName = "import_template";
			List<VDSGroup> clusters = DataProvider.GetClusterListByStorageDomain(Entity.id);

			model.Cluster.Items = clusters;
			model.Cluster.SelectedItem = Linq.FirstOrDefault(clusters);



			model.SourceStorage = Entity.StorageStaticData;
			model.StoragePool = DataProvider.GetFirstStoragePoolByStorageDomain(Entity.StorageStaticData.Id);

			//var destStorages = DataProvider.GetDataDomainsListByDomain(Entity.id)
			//    .Where(a => (a.storage_domain_type == StorageDomainType.Data || a.storage_domain_type == StorageDomainType.Master)
			//                && a.status.HasValue && a.status.Value == StorageDomainStatus.Active)
			//    .ToList();

			List<storage_domains> destStorages = new List<storage_domains>();
			foreach (storage_domains domain in DataProvider.GetDataDomainsListByDomain(Entity.id))
			{
				if ((domain.storage_domain_type == StorageDomainType.Data || domain.storage_domain_type == StorageDomainType.Master)
							&& domain.status.HasValue && domain.status.Value == StorageDomainStatus.Active)
				{
					destStorages.Add(domain);
				}
			}


			model.DestinationStorage.Items = destStorages;
			model.DestinationStorage.SelectedItem = Linq.FirstOrDefault(destStorages);

			model.Items = SelectedItems;

			if (destStorages.Count == 0)
			{
				model.DestinationStorage.IsChangable = false;
				model.DestinationStorage.ChangeProhibitionReasons.Add("Cannot import Template.");

				model.Message = "There is no Data Storage Domain to import the Template into. Please attach a Data Storage Domain to the Template's Data Center.";

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				model.Commands.Add(
					new UICommand("OnRestore", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		private void OnRestore()
		{
			ImportTemplateModel model = (ImportTemplateModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			//List<VdcReturnValueBase> ret = Frontend.RunMultipleActions(VdcActionType.ImportVmTemplate,
			//	SelectedItems.Cast<KeyValuePair<VmTemplate, List<DiskImage>>>()
			//	.Select(a => (VdcActionParametersBase)new ImprotVmTemplateParameters(model.StoragePool.id,
			//		model.SourceStorage.id,
			//		model.DestinationStorage.ValueAs<storage_domains>().id,
			//		model.Cluster.ValueAs<VDSGroup>().ID,
			//		a.Key)
			//	)
			//	.ToList()
			//);
			List<VdcActionParametersBase> prms = new List<VdcActionParametersBase>();
			foreach (object a in SelectedItems)
			{
				KeyValuePair<VmTemplate, List<DiskImage>> item = (KeyValuePair<VmTemplate, List<DiskImage>>)a;
				prms.Add(new ImprotVmTemplateParameters(model.StoragePool.Id,
														model.SourceStorage.Id,
														((storage_domains)model.DestinationStorage.SelectedItem).id,
														((VDSGroup)model.Cluster.SelectedItem).ID,
														item.Key));
			}

			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.ImportVmTemplate, prms,
				result =>
					{
						TemplateBackupModel templateBackupModel = (TemplateBackupModel) result.State;
						templateBackupModel.Window.StopProgress();
						templateBackupModel.Cancel();
						List<VdcReturnValueBase> retVals = (List<VdcReturnValueBase>)result.ReturnValue;
						if (retVals != null && templateBackupModel.SelectedItems.Count > retVals.Count)
						{
							ConfirmationModel confirmModel = new ConfirmationModel();
							templateBackupModel.ConfirmWindow = confirmModel;
							confirmModel.Title = "Import Template(s)";
							confirmModel.HashName = "import_template";

							string importedTemplates = String.Empty;
							int i = 0;
							foreach (object a in templateBackupModel.SelectedItems)
							{
								KeyValuePair<VmTemplate, List<DiskImage>> item = (KeyValuePair<VmTemplate, List<DiskImage>>)a;
								VmTemplate template = item.Key;
								if (Linq.FindVdcReturnValueByDescription(retVals, template.name) == null)
								{
									importedTemplates += template.name + (++i != templateBackupModel.SelectedItems.Count ? ", " : "");
								}
							}

							importedTemplates.Trim().TrimEnd(',');
							confirmModel.Message = string.Format("Import process has begun for Template(s): {0}.\nYou can check import status in the 'Events' tab of the specific destination storage domain, or in the main 'Events' tab", importedTemplates);

							confirmModel.Commands.Add(
								new UICommand("CancelConfirm", templateBackupModel)
								{
									Title = "Close",
									IsDefault = true,
									IsCancel = true
								});
						}
					}, this
				);
			

			
		}

		protected override void EntityPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.EntityPropertyChanged(sender, e);

			if (e.PropertyName == "storage_domain_shared_status")
			{
				SearchCommand.Execute();
			}
		}

		protected override void SyncSearch()
		{
			base.SyncSearch();

			if (Entity == null || Entity.storage_domain_type != StorageDomainType.ImportExport || Entity.storage_domain_shared_status != StorageDomainSharedStatus.Active)
			{
				Items = null;
				IsRefreshing = false;
			}
			else
			{
				IsRefreshing = true;
				Task.Create(this, 1).Run();
			}
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();
			SyncSearch();
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if (command.Name == "OnRestore")
			{
				OnRestore();
			}
		}

		private VdcQueryReturnValue returnValue;
		Guid storageDomainId = Guid.Empty;

		public void run(TaskContext context)
		{
			if (Entity != null)
			{
				switch ((int)context.State)
				{
					case 1:
						storageDomainId = Entity.id;

						storage_pool pool = DataProvider.GetFirstStoragePoolByStorageDomain(storageDomainId);

						returnValue = Frontend.RunQuery(VdcQueryType.GetTemplatesFromExportDomain,
							new GetAllFromExportDomainQueryParamenters(pool.Id, storageDomainId)
							{
								GetAll = true
							});

						context.InvokeUIThread(this, 2);
						break;

					case 2:
						//if user didn't change the entity meanwhile, update Items, else dont touch it
						if (storageDomainId.Equals(Entity.id))
						{
							if (returnValue != null && returnValue.Succeeded)
							{
								//Items = ((Dictionary<VmTemplate, List<DiskImage>>)returnValue.ReturnValue).ToList();
								List<KeyValuePair<VmTemplate, List<DiskImage>>> items = new List<KeyValuePair<VmTemplate, List<DiskImage>>>();
								Dictionary<VmTemplate, List<DiskImage>> list = (Dictionary<VmTemplate, List<DiskImage>>)returnValue.ReturnValue;

								foreach (KeyValuePair<VmTemplate, List<DiskImage>> item in list)
								{
									items.Add(item);
								}
								Items = items;
							}
							else
							{
								Items = new List<KeyValuePair<VmTemplate, List<DiskImage>>>();
							}
							IsRefreshing = false;
						}
						returnValue = null;
						storageDomainId = Guid.Empty;
						break;
				}
			}
		}
	}
}
