using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using System.Windows;
using System.Windows.Threading;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;

namespace org.ovirt.engine.ui.uicommon.models.tags
{
	public class TagListModel : SearchableListModel
	{
		#region Commands

		public UICommand NewCommand { get; private set; }
		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }
		public UICommand ResetCommand { get; private set; }

		#endregion

		#region Properties

		public new TagModel SelectedItem
		{
			get { return (TagModel)base.SelectedItem; }
			set { base.SelectedItem = value; }
		}

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		public IDictionary<Guid, bool> AttachedTagsToEntities { get; set; }

		#endregion

		public TagListModel()
		{
			NewCommand = new UICommand("New", this);
			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);
			ResetCommand = new UICommand("Reset", this);

			SearchCommand.Execute();

			UpdateActionAvailability();

			//Initialize SelectedItems property with empty collection.
			SelectedItems = new List<TagModel>();
		}

		protected override void SyncSearch()
		{
			base.SyncSearch();

			TagModel rootTag = TagToModel(DataProvider.GetRootTag());
			rootTag.Name.Entity = "Root";
			rootTag.Type = TagModelType.Root;
			rootTag.IsChangable = false;
			Items = new List<TagModel> { rootTag };
		}

		protected override void ItemsChanged()
		{
			base.ItemsChanged();
			if (AttachedTagsToEntities != null)
			{
				List<TagModel> tags = (List<TagModel>)Items;
				RecursiveSetSelection(tags[0], AttachedTagsToEntities);
			}
		}

		public void RecursiveSetSelection(TagModel tagModel, IDictionary<Guid, bool> attachedEntities)
		{
			if (attachedEntities.ContainsKey(tagModel.Id) && attachedEntities[tagModel.Id])
			{
				tagModel.Selection = true;
			}
			else
			{
				tagModel.Selection = false;
			}
			if (tagModel.Children != null)
			{
				foreach (TagModel subModel in tagModel.Children)
				{
					RecursiveSetSelection(subModel, attachedEntities);
				}
			}
		}

		private TagModel TagToModel(VdcCommon.BusinessEntities.tags tag)
		{
			EntityModel name = new EntityModel { Entity = tag.tag_name };
			EntityModel description = new EntityModel { Entity = tag.description };

			List<TagModel> children = new List<TagModel>();
			foreach (VdcCommon.BusinessEntities.tags a in tag.Children)
			{
				children.Add(TagToModel(a));
			}

			TagModel model = new TagModel();
			model.Id = tag.tag_id;
			model.Name = name;
			model.Description = description;
			model.Type = tag.IsReadonly.GetValueOrDefault() ? TagModelType.ReadOnly : TagModelType.Regular;
			model.Selection = false;
			model.ParentId = tag.parent_id == null ? Guid.Empty : tag.parent_id.Value;
			model.Children = children;

			model.SelectionChangedEvent.addListener(this);

			return model;
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(TagModel.SelectionChangedEventDefinition))
			{
				OnTagSelectionChanged(sender, args);
			}
		}

		private void OnTagSelectionChanged(object sender, EventArgs e)
		{
			TagModel model = (TagModel)sender;

			List<TagModel> list = new List<TagModel>();
			if (SelectedItems != null)
			{
				foreach (object item in SelectedItems)
				{
					list.Add((TagModel)item);
				}
			}

			if (model.Selection.GetValueOrDefault())
			{
				list.Add(model);
			}
			else
			{
				list.Remove(model);
			}

			SelectedItems = list;
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();
			SyncSearch();
		}

		private void Reset()
		{
			SelectedItems = new List<TagModel>();

			if (Items != null)
			{
				foreach (object item in Items)
				{
					ResetInternal((TagModel) item);
				}
			}
		}

		private void ResetInternal(TagModel root)
		{
			root.Selection = false;
			foreach (TagModel item in root.Children)
			{
				ResetInternal(item);
			}
		}

		public void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Tag(s)";
			model.HashName = "remove_tag";
			model.Message = "Tag(s):";
			
			List<object> items = new List<object>();
			items.Add(SelectedItem.Name.Entity);
			model.Items = items;

			model.Note = "NOTE:\n  - Removing the tag will also remove all of its descendants.\n  - Tag and descendants will be erased from all objects that are attached to them.";

			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnRemove()
		{
			VdcReturnValueBase returnValue = Frontend.RunAction(VdcActionType.RemoveTag, new TagsActionParametersBase(SelectedItem.Id));
			if (returnValue != null && returnValue.Succeeded)
			{
				SearchCommand.Execute();
			}

			Cancel();
		}

		public void Edit()
		{
			if (Window != null)
			{
				return;
			}

			TagModel model = new TagModel();
			Window = model;
			model.Title = "Edit Tag";
			model.HashName = "edit_tag";
			model.IsNew = false;
			model.Name.Entity = SelectedItem.Name.Entity;
			model.Description.Entity = SelectedItem.Description.Entity;
			model.ParentId = SelectedItem.ParentId;

			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void New()
		{
			if (Window != null)
			{
				return;
			}

			TagModel model = new TagModel();
			Window = model;
			model.Title = "New Tag";
			model.HashName = "new_tag";
			model.IsNew = true;

			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnSave()
		{
			TagModel model = (TagModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			VdcCommon.BusinessEntities.tags tag =
				new VdcCommon.BusinessEntities.tags
				{
					tag_id = model.IsNew ? Guid.Empty : SelectedItem.Id,
					parent_id = model.IsNew ? SelectedItem.Id : SelectedItem.ParentId,
					tag_name = (string)model.Name.Entity,
					description = (string)model.Description.Entity
				};


			model.StartProgress(null);

			Frontend.RunAction(model.IsNew ? VdcActionType.AddTag : VdcActionType.UpdateTag,
				new TagsOperationParameters(tag),
				result =>
				{
					TagListModel localModel = (TagListModel)result.State;

					localModel.PostOnSave(result.ReturnValue);
				},
				this
			);
		}

		public void PostOnSave(VdcReturnValueBase returnValue)
		{
			TagModel model = (TagModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();
				SearchCommand.Execute();
			}
		}

		public void Cancel()
		{
			Window = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		private void UpdateActionAvailability()
		{
			NewCommand.IsExecutionAllowed = SelectedItem != null;
			EditCommand.IsExecutionAllowed = SelectedItem != null && SelectedItem.Type == TagModelType.Regular;
			RemoveCommand.IsExecutionAllowed = SelectedItem != null && SelectedItem.Type == TagModelType.Regular;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == ResetCommand)
			{
				Reset();
			}
			else if (command == NewCommand)
			{
				New();
			}
			else if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
		}
	}
}
