using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.configure;
using org.ovirt.engine.ui.uicommon.models.vms;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using VdcCommon;
using System.ComponentModel;
using org.ovirt.engine.ui.uicommon.dataprovider;

namespace org.ovirt.engine.ui.uicommon.models.templates
{
	public class TemplateListModel : ListWithDetailsModel, ISupportSystemTreeContext
	{
		#region Commands

		public UICommand EditCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }
		public UICommand ExportCommand { get; private set; }
		public UICommand CopyCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		protected object[] SelectedKeys
		{
			//			get { return SelectedItems == null ? new object[0] : SelectedItems.Cast<VmTemplate>().Select(a => a.vmt_guid).Cast<object>().ToArray(); }
			get
			{
				if (SelectedItems == null)
				{
					return new object[0];
				}
				else
				{
					List<guid> items = new List<guid>();
					foreach (object item in SelectedItems)
					{
						VmTemplate a = (VmTemplate)item;
						items.Add(a.Id);
					}
					return items.ToArray();
				}
			}
		}

		private SystemTreeItemModel systemTreeSelectedItem;
		public SystemTreeItemModel SystemTreeSelectedItem
		{
			get { return systemTreeSelectedItem; }
			set
			{
				if (systemTreeSelectedItem != value)
				{
					systemTreeSelectedItem = value;
				}
			}
		}
		#endregion


		public TemplateListModel()
		{
			Title = "Templates";

			DefaultSearchString = "Template:";
			SearchString = DefaultSearchString;

			EditCommand = new UICommand("Edit", this);
			RemoveCommand = new UICommand("Remove", this);
			ExportCommand = new UICommand("Export", this);
			CopyCommand = new UICommand("Copy", this);

			UpdateActionAvailability();

			SearchNextPageCommand.IsAvailable = true;
			SearchPreviousPageCommand.IsAvailable = true;
		}

		private void Copy()
		{
			VmTemplate template = (VmTemplate)SelectedItem;

			if (Window != null)
			{
				return;
			}

			ListModel model = new ListModel();
			Window = model;
			model.Title = "Copy Template";
			model.HashName = "copy_template";

			//Select all active data storage domains where the template is not copied to.
			List<storage_domains> domainsWithTemplate = DataProvider.GetStorageDomainListByTemplate(template.Id);
			Guid storagePoolId = template.storage_pool_id != null ? template.storage_pool_id.Value : Guid.Empty;
			List<storage_domains> list = DataProvider.GetStorageDomainList(storagePoolId);
			list.Sort(new Linq.StorageDomainByNameComparer());

			List<EntityModel> items = new List<EntityModel>();
			bool isTemplateExistInOneActiveDomain = false;
			foreach (storage_domains a in list)
			{
				bool templateNotExistInAnyDomain_iter = true;
				foreach (storage_domains b in domainsWithTemplate)
				{
					if (b.id.Equals(a.id) && b.status.GetValueOrDefault(StorageDomainStatus.InActive) == StorageDomainStatus.Active)
					{
						templateNotExistInAnyDomain_iter = false;
						isTemplateExistInOneActiveDomain = true;
						break;
					}
				}

				if ((a.storage_domain_type == StorageDomainType.Data || a.storage_domain_type == StorageDomainType.Master)
						&& templateNotExistInAnyDomain_iter
						&& a.status.GetValueOrDefault() == StorageDomainStatus.Active)
				{
					EntityModel entityModel = new EntityModel();
					entityModel.Entity = a;

					items.Add(entityModel);
				}
			}

			model.Items = items;

			if (items.Count == 1)
			{
				items[0].IsSelected = true;
			}

			if (items.Count == 0)
			{
				if (isTemplateExistInOneActiveDomain)
				{
					model.Message = "Template already exists on all available Storage Domains.";
				}
				else
				{
					model.Message = "No Storage Domain is available - check Storage Domains and Hosts status.";
				}

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				model.Commands.Add(
					new UICommand("OnCopy", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		private void OnCopy()
		{
			VmTemplate template = (VmTemplate)SelectedItem;
			ListModel model = (ListModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> items = new List<VdcActionParametersBase>();
			foreach (object item in model.Items)
			{
				EntityModel a = (EntityModel)item;
				if (a.IsSelected)
				{
					items.Add(new MoveOrCopyParameters(template.Id, ((storage_domains)a.Entity).id));
				}
			}

			//should be only 1
			if (items.Count == 0)
			{
				return;
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.MoveOrCopyTemplate, items,
				result =>
				{
					ListModel localModel = (ListModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}


		private void Export()
		{
			VmTemplate template = (VmTemplate)SelectedItem;

			if (Window != null)
			{
				return;
			}

			ExportVmModel model = new ExportVmModel();
			Window = model;
			model.Title = "Backup Template";
			model.HashName = "backup_template";

			// 			var storages = DataProvider.GetStorageDomainList(template.storage_pool_id.Value)
			// 				.Where(a => a.storage_domain_type == StorageDomainType.ImportExport)
			// 				.ToList();
			List<storage_domains> storages = new List<storage_domains>();
			List<storage_domains> domains = DataProvider.GetStorageDomainList(template.storage_pool_id.Value);
			foreach (storage_domains a in domains)
			{
				if (a.storage_domain_type == StorageDomainType.ImportExport)
				{
					storages.Add(a);
				}
			}

			model.Storage.Items = storages;
			model.Storage.SelectedItem = Linq.FirstOrDefault(storages);


			bool isAllStoragesActive = true;
			foreach (storage_domains a in storages)
			{
				if (a.status != StorageDomainStatus.Active)
				{
					isAllStoragesActive = false;
					break;
				}
			}

			if (SelectedTemplatesOnDifferentDataCenters())
			{
				model.CollapseSnapshots.IsChangable = false;
				model.ForceOverride.IsChangable = false;

				model.Message = "Templates reside on several Data Centers. Make sure the exported Templates reside on the same Data Center.";

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}

			else if (storages.Count == 0)
			{
				model.ForceOverride.IsChangable = false;

				model.Message = "There is no Export Domain to export the Template into. Please attach an Export Domain to the Template's Data Center.";

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else if (!isAllStoragesActive)
			{
				model.ForceOverride.IsChangable = false;

				model.Message = "The relevant Export Domain in not active. Please activate it.";

				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				showWarningOnExistingTemplates(model);

				model.Commands.Add(
					new UICommand("OnExport", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		private bool SelectedTemplatesOnDifferentDataCenters()
		{
			List<VmTemplate> templates = Linq.Cast<VmTemplate>(SelectedItems);
			//return templates.GroupBy(a => a.storage_pool_id).Count() > 1 ? true : false;

			IDictionary<nguid, List<VmTemplate>> t = new Dictionary<nguid, List<VmTemplate>>();
			foreach (VmTemplate a in templates)
			{
				if (!a.Id.Equals(Guid.Empty))
				{
					if (!t.ContainsKey(a.storage_pool_id))
					{
						t.Add(a.storage_pool_id, new List<VmTemplate>());
					}

					List<VmTemplate> list = t[a.storage_pool_id];
					list.Add(a);
				}
			}

			return t.Count > 1 ? true : false;
		}

		private void showWarningOnExistingTemplates(ExportVmModel model)
		{
			Guid storageDomainId = ((storage_domains)model.Storage.SelectedItem).id;
			storage_pool storagePool = DataProvider.GetFirstStoragePoolByStorageDomain(storageDomainId);
			string existingTemplates = string.Empty;
			if (storagePool != null)
			{
				VdcQueryReturnValue returnValue = Frontend.RunQuery(VdcQueryType.GetTemplatesFromExportDomain,
									new GetAllFromExportDomainQueryParamenters(storagePool.Id, storageDomainId) { GetAll = true });

				if (returnValue != null && returnValue.Succeeded && returnValue.ReturnValue != null)
				{
					//foreach (VmTemplate template in SelectedItems.Cast<VmTemplate>())
					foreach (VmTemplate template in Linq.Cast<VmTemplate>(SelectedItems))
					{
						bool found = false;
						Dictionary<VmTemplate, List<DiskImage>> items = (Dictionary<VmTemplate, List<DiskImage>>)returnValue.ReturnValue;

						foreach (VmTemplate a in items.Keys)
						{
							if (a.Id.Equals(template.Id))
							{
								found = true;
								break;
							}
						}

						//if (((Dictionary<VmTemplate, List<DiskImage>>)returnValue.ReturnValue).Keys.SingleOrDefault(a => a.vmt_guid == template.vmt_guid) != null)
						if (found)
						{
							existingTemplates += "\u2022    " + template.name + "\n";
						}
					}
				}
				if (!String.IsNullOrEmpty(existingTemplates))
				{
					model.Message = String.Format("Template(s):\n{0} already exist on the target Export Domain. If you want to override them, please check the 'Force Override' check-box.", existingTemplates);
				}
			}
		}

		private void OnExport()
		{
			ExportVmModel model = (ExportVmModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VmTemplate a = (VmTemplate)item;
				if(a.Id.Equals(Guid.Empty))
				{
					continue;
				}
				list.Add(new MoveOrCopyParameters(a.Id, ((storage_domains)model.Storage.SelectedItem).id) { ForceOverride = (bool)model.ForceOverride.Entity });
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.ExportVmTemplate, list,
				result =>
				{
					ExportVmModel localModel = (ExportVmModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		protected override void InitDetailModels()
		{
			base.InitDetailModels();

			ObservableCollection<EntityModel> list = new ObservableCollection<EntityModel>();
			list.Add(new TemplateGeneralModel());
			list.Add(new TemplateVmListModel());
			list.Add(new TemplateInterfaceListModel());
			list.Add(new TemplateDiskListModel());
			list.Add(new TemplateStorageListModel());
			list.Add(new TemplateEventListModel());
			list.Add(new PermissionListModel());
			DetailModels = list;
		}

		public override bool IsSearchStringMatch(string searchString)
		{
			return searchString.Trim().ToLower().StartsWith("template");
		}

		protected override void SyncSearch()
		{
			base.SyncSearch(VdcQueryType.Search, new SearchParameters(SearchString, SearchType.VmTemplate)
			{
				MaxCount = SearchPageSize
			});
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			AsyncResult = Frontend.RegisterSearch(SearchString, SearchType.VmTemplate, SearchPageSize);
			Items = AsyncResult.Data;
		}

		private void Edit()
		{
			VmTemplate template = (VmTemplate)SelectedItem;

			if (Window != null)
			{
				return;
			}

			UnitVmModel model = new UnitVmModel(new TemplateVmModelBehavior(template));
			Window = model;
			model.Title = "Edit Template";
			model.HashName = "edit_template";
			model.VmType = template.vm_type;

			model.Initialize(this.SystemTreeSelectedItem);


			model.Commands.Add(
				new UICommand("OnSave", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void Remove()
		{
			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Template(s)";
			model.HashName = "remove_template";
			model.Message = "Template(s)";

			List<string> items = new List<string>();
			List<VmTemplate> templates = Linq.Cast<VmTemplate>(SelectedItems);
			foreach (VmTemplate template in templates)
			{
				if (!template.Id.Equals(Guid.Empty))
				{
					items.Add(template.name);
				}
			}

			model.Items = items;


			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnRemove()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VmTemplate a = (VmTemplate)item;
				list.Add(new VmTemplateParametersBase(a.Id));
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.RemoveVmTemplate, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
				},
				model
			);
		}

		private void OnSave()
		{
			UnitVmModel model = (UnitVmModel)Window;

			if (!model.Validate())
			{
				return;
			}

			string name = (string)model.Name.Entity;

			AsyncDataProvider.IsTemplateNameUnique(new AsyncQuery(this,
				(target, returnValue) =>
				{
					TemplateListModel templateListModel = (TemplateListModel)target;
					bool isNameUnique = (bool)returnValue;
					templateListModel.PostNameUniqueCheck(isNameUnique);
				})
				, name
			);
		}

		public void PostNameUniqueCheck(bool isNameUnique)
		{
			UnitVmModel model = (UnitVmModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			VmTemplate selectedItem = (VmTemplate)SelectedItem;
			VmTemplate template = (VmTemplate)Cloner.Clone(selectedItem);

			string name = (string)model.Name.Entity;

			//Check name unicitate.
			if (!isNameUnique && String.Compare(name, template.name, true) != 0)
			{
				model.Name.InvalidityReasons.Add("Name must be unique.");
				model.Name.IsValid = false;
				model.IsGeneralTabValid = false;
				return;
			}

			//Save changes.
			template.vm_type = model.VmType;
			template.name = name;
			template.os = (VmOsType)model.OSType.SelectedItem;
			template.num_of_monitors = (int)model.NumOfMonitors.SelectedItem;
			template.description = (string)model.Description.Entity;
			template.domain = model.Domain.IsAvailable ? (string)model.Domain.SelectedItem : String.Empty;
			template.mem_size_mb = (int)model.MemSize.Entity;
			template.vds_group_id = ((VDSGroup)model.Cluster.SelectedItem).ID;
			template.time_zone = (model.TimeZone.IsAvailable && model.TimeZone.SelectedItem != null) ? ((KeyValuePair<string, string>)model.TimeZone.SelectedItem).Key : String.Empty;
			template.num_of_sockets = (int)model.NumOfSockets.Entity;
			template.cpu_per_socket = (int)model.TotalCPUCores.Entity / (int)model.NumOfSockets.Entity;
			template.usb_policy = (UsbPolicy)model.UsbPolicy.SelectedItem;
			template.is_auto_suspend = false;
			template.is_stateless = (bool)model.IsStateless.Entity;
			template.default_boot_sequence = model.BootSequence;
			template.iso_path = model.CdImage.IsChangable ? (string)model.CdImage.SelectedItem : String.Empty;
			template.auto_startup = (bool)model.IsHighlyAvailable.Entity;
			template.kernel_url = (string)model.Kernel_path.Entity;
			template.kernel_params = (string)model.Kernel_parameters.Entity;
			template.initrd_url = (string)model.Initrd_path.Entity;


			EntityModel displayProtocolSelectedItem = (EntityModel)model.DisplayProtocol.SelectedItem;
			template.default_display_type = (DisplayType)displayProtocolSelectedItem.Entity;

			EntityModel prioritySelectedItem = (EntityModel)model.Priority.SelectedItem;
			template.priority = (int)prioritySelectedItem.Entity;

			model.StartProgress(null);

			Frontend.RunAction(VdcActionType.UpdateVmTemplate,
				new UpdateVmTemplateParameters(template),
				result =>
				{
					TemplateListModel localModel = (TemplateListModel)result.State;
					localModel.PostUpdateVmTemplate(result.ReturnValue);
				},
				this
			);
		}

		public void PostUpdateVmTemplate(VdcReturnValueBase returnValue)
		{
			UnitVmModel model = (UnitVmModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();
			}
		}

		private void Cancel()
		{
			Frontend.Unsubscribe();

			Window = null;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void SelectedItemPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.SelectedItemPropertyChanged(sender, e);

			if (e.PropertyName == "status")
			{
				UpdateActionAvailability();
			}
		}

		private bool SelectedItemsContainBlankTemplate()
		{
			if (SelectedItems != null)
			{
				List<VmTemplate> templates = Linq.Cast<VmTemplate>(SelectedItems);
				foreach (VmTemplate template in templates)
				{
					if (template != null && template.Id.Equals(Guid.Empty))
					{
						return true;
					}
				}
			}

			return false;
		}

		protected virtual void UpdateActionAvailability()
		{
			VmTemplate item = (VmTemplate)SelectedItem;
			ArrayList items = (ArrayList)SelectedItems ?? new ArrayList();

			bool blankSelected = SelectedItem != null && item.Id == Guid.Empty; 

			EditCommand.IsExecutionAllowed = items.Count == 1
				&& item != null
				&& item.status != VmTemplateStatus.Locked;
			if (EditCommand.IsExecutionAllowed && blankSelected)
			{
				EditCommand.IsExecutionAllowed = false;
				EditCommand.ExecuteProhibitionReasons.Add("Blank Template cannot be edited");
			}

			RemoveCommand.IsExecutionAllowed = items.Count > 0
			                                   && VdcActionUtils.CanExecute(items, typeof (VmTemplate),
			                                                             VdcActionType.RemoveVmTemplate);
			if(RemoveCommand.IsExecutionAllowed && blankSelected)
			{
				RemoveCommand.IsExecutionAllowed = false;
				RemoveCommand.ExecuteProhibitionReasons.Add("Blank Template cannot be removed");
			}

			ExportCommand.IsExecutionAllowed = items.Count > 0
			                                   && VdcActionUtils.CanExecute(items, typeof (VmTemplate),
			                                                             VdcActionType.ExportVmTemplate);

			if (ExportCommand.IsExecutionAllowed && blankSelected)
			{
				ExportCommand.IsExecutionAllowed = false;
				ExportCommand.ExecuteProhibitionReasons.Add("Blank Template cannot be exported");
			}

			CopyCommand.IsExecutionAllowed = items.Count == 1
			                                 && item != null
			                                 && VdcActionUtils.CanExecute(items, typeof (VmTemplate),
			                                                           VdcActionType.MoveOrCopyTemplate);
			
			if (CopyCommand.IsExecutionAllowed && blankSelected)
			{
				CopyCommand.IsExecutionAllowed = false;
				CopyCommand.ExecuteProhibitionReasons.Add("Blank Template cannot be copied");
			}
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == EditCommand)
			{
				Edit();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command == CopyCommand)
			{
				Copy();
			}
			else if (command == ExportCommand)
			{
				Export();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnCopy")
			{
				OnCopy();
			}
			else if (command.Name == "OnExport")
			{
				OnExport();
			}
			else if (command.Name == "OnSave")
			{
				OnSave();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
		}
	}
}
