using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.dataprovider;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using System.ComponentModel;
using org.ovirt.engine.ui.uicompat;
using VdcCommon;

namespace org.ovirt.engine.ui.uicommon.models.templates
{
	public class TemplateStorageListModel : SearchableListModel
	{
		#region Commands

		public UICommand RemoveCommand { get; private set; }

		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		#endregion

		public TemplateStorageListModel()
		{
			Title = "Storage";

			RemoveCommand = new UICommand("Remove", this);

			UpdateActionAvailability();
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();

			if (Entity != null)
			{
				SearchCommand.Execute();
			}

			UpdateActionAvailability();
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();

			VmTemplate template = (VmTemplate)Entity;

			AsyncResult = Frontend.RegisterQuery(VdcQueryType.GetStorageDomainsByVmTemplateId, new GetStorageDomainsByVmTemplateIdQueryParameters(template.Id));
			Items = AsyncResult.Data;
		}

		protected override void SyncSearch()
		{
			base.SyncSearch();

			VmTemplate template = (VmTemplate)Entity;
			base.SyncSearch(VdcQueryType.GetStorageDomainsByVmTemplateId, new GetStorageDomainsByVmTemplateIdQueryParameters(template.Id));
		}

		private void Remove()
		{
			VmTemplate template = (VmTemplate)Entity;

			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Template from Storage Domain";
			model.HashName  = "remove_template_from_storage_domains";
			model.Message = String.Format("Are you sure you want to remove the Template {0} from the following Storage Domain(s)?", template.name);

			//Show warning if template is going to be removed from all storage domains it exist on.
			if (SelectedItems.Count == ((IList)Items).Count)
			{
				model.Note = "Note: This action will remove the Template permanently from all Storage Domains.";
			}

			List<string> items = new List<string>();
			foreach (object item in SelectedItems)
			{
				storage_domains a = (storage_domains)item;
				items.Add(a.storage_name);
			}
			model.Items = items;

			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		private void OnRemove()
		{
			VmTemplate template = (VmTemplate)Entity;

			List<Guid> ids = new List<Guid>();
			foreach (object item in SelectedItems)
			{
				storage_domains a = (storage_domains)item;
				ids.Add(a.id);
			}

			Frontend.RunActionAsyncroniousely(VdcActionType.RemoveVmTemplate,
				new VmTemplateParametersBase(template.Id)
				{
					StorageDomainsList = ids
				});

			Cancel();
		}

		private void Cancel()
		{
			Window = null;
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void EntityPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.EntityPropertyChanged(sender, e);

			if (e.PropertyName == "status")
			{
				UpdateActionAvailability();
			}
		}

		private void UpdateActionAvailability()
		{
			VmTemplate template = (VmTemplate)Entity;
			List<storage_domains> selectedItems = SelectedItems != null ? Linq.Cast<storage_domains>(SelectedItems) : new List<storage_domains>();

			RemoveCommand.IsExecutionAllowed = template != null
				&& template.status == VmTemplateStatus.OK
				&& selectedItems.Count > 0;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
		}
	}
}
