using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Text;
using System.Windows;
using org.ovirt.engine.ui.uicommon.dataprovider;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.Interfaces;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using System.Collections;

namespace org.ovirt.engine.ui.uicommon.models.vms
{
	public class VmSnapshotListModel : ListModel
	{
		#region Commands

		public UICommand NewCommand { get; private set; }
		public UICommand PreviewCommand { get; private set; }
		public UICommand CommitCommand { get; private set; }
		public UICommand UndoCommand { get; private set; }
		public UICommand RemoveCommand { get; private set; }
		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		public new SnapshotModel SelectedItem
		{
			get { return base.SelectedItem as SnapshotModel; }
			set { base.SelectedItem = value; }
		}

		private IEnumerable apps;
		public IEnumerable Apps
		{
			get { return apps; }
			set
			{
				if (apps != value)
				{
					apps = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Apps"));
				}
			}
		}

		private bool isSnapshotsAvailable;
		public bool IsSnapshotsAvailable
		{
			get { return isSnapshotsAvailable; }
			set
			{
				if (isSnapshotsAvailable != value)
				{
					isSnapshotsAvailable = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsSnapshotsAvailable"));
				}
			}
		}

		public EntityModel CanSelectSnapshot { get; private set; }

		#endregion

		public VmSnapshotListModel()
		{
			Title = "Snapshots";

			NewCommand = new UICommand("New", this);
			PreviewCommand = new UICommand("Preview", this);
			CommitCommand = new UICommand("Commit", this);
			UndoCommand = new UICommand("Undo", this);
			RemoveCommand = new UICommand("Remove", this);

			CanSelectSnapshot = new EntityModel();
		}

		private void Remove()
		{
			if (Entity != null)
			{
				if (Window != null)
				{
					return;
				}

				ConfirmationModel model = new ConfirmationModel();
				Window = model;
				model.Title = "Delete Snapshot";
				model.HashName = "delete_snapshot";
				model.Message = String.Format("Are you sure you want to delete snapshot from {0} with description '{1}'?", SelectedItem.Date, SelectedItem.Description.Entity);

				model.Commands.Add(
					new UICommand("OnRemove", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		private void OnRemove()
		{
			if (SelectedItem == null)
			{
				Cancel();
				return;
			}

			VM vm = (VM)Entity;
			if (vm != null)
			{
				foreach (DiskImage disk in data.Keys)
				{
					//var snapshots = data[disk].OrderBy(a => a.lastModified);

					List<DiskImage> list = new List<DiskImage>();
					foreach (DiskImage a in data[disk])
					{
						list.Add(a);
					}
					list.Sort(new Linq.DiskImageByLastModifiedComparer());

					if (list.Count < 2)
					{
						continue;
					}

					//var srcSnapshot = snapshots.FirstOrDefault(a => a.vm_snapshot_id == SelectedItem.SnapshotId);
					DiskImage srcSnapshot = null;
					foreach (DiskImage a in list)
					{
						if (a.vm_snapshot_id.Equals(SelectedItem.SnapshotId))
						{
							srcSnapshot = a;
							break;
						}
					}
					if (srcSnapshot == null)
					{
						continue;
					}

					//var dstSnapshot = snapshots.FirstOrDefault(a => a.ParentId == srcSnapshot.image_guid);
					DiskImage dstSnapshot = null;
					foreach (DiskImage a in list)
					{
						if (a.ParentId.Equals(srcSnapshot.Id))
						{
							dstSnapshot = a;
							break;
						}
					}
					if (dstSnapshot == null)
					{
						continue;
					}

					Guid srcSnapshotId = (srcSnapshot.vm_snapshot_id != null) ? srcSnapshot.vm_snapshot_id.Value : Guid.Empty;
					Guid dstSnapshotId = (dstSnapshot.vm_snapshot_id != null) ? dstSnapshot.vm_snapshot_id.Value : Guid.Empty;

					Frontend.RunAction(VdcActionType.MergeSnapshot,
						new MergeSnapshotParamenters(srcSnapshotId, dstSnapshotId, vm.vm_guid),
						result =>
						{
						},
						null
					);

					CanSelectSnapshot.Entity = false;

					break;
				}
			}

			Cancel();
		}

		private void Undo()
		{
			VM vm = (VM)Entity;
			if (vm != null)
			{
				Guid snapshotId = Guid.Empty;

				foreach (DiskImage disk in data.Keys)
				{
					//Find the last snapshot to revert to it.
					//var snapshot = data[disk]
					//    //.OrderByDescending(a => a.lastModified)
					//    .FirstOrDefault();

					DiskImage snapshot = data[disk][0];
					if (snapshot != null)
					{
						snapshotId = (snapshot.vm_snapshot_id != null) ? snapshot.vm_snapshot_id.Value : Guid.Empty;
						break;
					}
				}

				if (!snapshotId.Equals(Guid.Empty))
				{
					Frontend.RunAction(VdcActionType.RestoreAllSnapshots,
						new RestoreAllSnapshotsParameters(vm.vm_guid, snapshotId),
						result =>
						{
						},
						null
					);
				}
			}
		}

		private void Commit()
		{
			VM vm = (VM)Entity;
			if (vm != null)
			{
				Guid snapshotId = Guid.Empty;

				foreach (DiskImage disk in data.Keys)
				{
					//Find a previwing snapshot by disk.
					//var snapshot = data[disk].FirstOrDefault(a => (Guid)a.image_guid == previewingImages[disk.image_guid]);
					DiskImage snapshot = null;
					foreach (DiskImage a in data[disk])
					{
						if (a.Id.Equals(previewingImages[disk.Id]))
						{
							snapshot = a;
							break;
						}
					}

					if (snapshot != null)
					{
						snapshotId = (snapshot.vm_snapshot_id != null) ? snapshot.vm_snapshot_id.Value : Guid.Empty;
						break;
					}
				}

				if (!snapshotId.Equals(Guid.Empty))
				{
					Frontend.RunAction(VdcActionType.RestoreAllSnapshots,
						new RestoreAllSnapshotsParameters(vm.vm_guid, snapshotId),
						result =>
						{
						},
						null
					);
				}
			}
		}

		private void Preview()
		{
			VM vm = (VM)Entity;
			if (vm != null)
			{
				Frontend.RunAction(VdcActionType.TryBackToAllSnapshotsOfVm,
					new TryBackToAllSnapshotsOfVmParameters(vm.vm_guid, SelectedItem.SnapshotId),
					result =>
					{
					},
					null
				);
			}
		}

		private void New()
		{
			if (Entity != null)
			{
				if (Window != null)
				{
					return;
				}

				SnapshotModel model = new SnapshotModel();
				Window = model;
				model.Title = "Create Snapshot";
				model.HashName = "create_snapshot";

				//var disks = data.Keys
				//    .Select(a => String.Format("Disk {0}", a.internal_drive_mapping))
				//    .OrderBy(a => a)
				//    .Select(a => new EntityModel() { Entity = a })
				//    .ToList();

				List<string> driveMappings = new List<string>();
				foreach (DiskImage a in data.Keys)
				{
					driveMappings.Add(String.Format("Disk {0}", a.internal_drive_mapping));
				}
				driveMappings.Sort();

				List<EntityModel> disks = new List<EntityModel>();
				foreach (string a in driveMappings)
				{
					EntityModel m = new EntityModel();
					m.Entity = String.Format("Disk {0}", a);
					disks.Add(m);
				}
				model.Disks = disks;

				//disks.Each(a => Selector.SetIsSelected(a, true));
				foreach (EntityModel a in disks)
				{
					a.IsSelected = true;
				}


				if (disks.Count == 0)
				{
					model.Message = "Snapshot cannot be created since the VM has no Disks";

					model.Commands.Add(
						new UICommand("Cancel", this)
						{
							Title = "Close",
							IsDefault = true,
							IsCancel = true
						});
				}
				else
				{
					model.Commands.Add(
						new UICommand("OnNew", this)
						{
							Title = "OK",
							IsDefault = true
						});
					model.Commands.Add(
						new UICommand("Cancel", this)
						{
							Title = "Cancel",
							IsCancel = true
						});
				}
			}
		}

		private void OnNew()
		{
			VM vm = (VM)Entity;
			if (vm == null)
			{
				return;
			}

			SnapshotModel model = (SnapshotModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			List<string> disks = new List<string>();
			foreach (EntityModel a in model.Disks)
			{
				if (a.IsSelected)
				{
					disks.Add(GetInternalDriveMapping((string)a.Entity));
				}
			}


			model.StartProgress(null);

			Frontend.RunAction(VdcActionType.CreateAllSnapshotsFromVm,
				new CreateAllSnapshotsFromVmParameters(vm.vm_guid, (string)model.Description.Entity)
				{
					DisksList = disks
				},
				result =>
				{
					VmSnapshotListModel localModel = (VmSnapshotListModel)result.State;

					localModel.PostOnNew(result.ReturnValue);
				},
				this
			);
		}

		public void PostOnNew(VdcReturnValueBase returnValue)
		{
			SnapshotModel model = (SnapshotModel)Window;

			model.StopProgress();

			if (returnValue != null && returnValue.Succeeded)
			{
				Cancel();
			}
		}

		/// <summary>
		/// Getting a string in the format of "Disk x", should return "x".
		/// *** NOTE: When localizing, this function should be smarter.
		/// </summary>
		/// <param name="diskDisplayName">string in the format of "Disk x"</param>
		/// <returns>"x"</returns>
		private string GetInternalDriveMapping(string diskDisplayName)
		{
			return diskDisplayName.Substring(5);
		}

		private void Cancel()
		{
			Window = null;
		}

		protected override void OnEntityChanged()
		{
			base.OnEntityChanged();

			//Deal with pool as Entity without failing.
			if (Entity is vm_pools)
			{
				IsSnapshotsAvailable = false;
				Items = null;
			}
			else
			{
				IsSnapshotsAvailable = true;
				UpdateData();
			}
		}

		private void UpdateData()
		{
			VM vm = (VM)Entity;
			if (vm == null)
			{
				return;
			}


			new GetAllVmSnapshotsExecutor(vm.vm_guid, new AsyncQuery(this,
				(target1, returnValue1) =>
				{
					VmSnapshotListModel model = (VmSnapshotListModel)target1;

					model.PostUpdateData(returnValue1);
				})
			)
			.Execute();
		}

		public void PostUpdateData(object returnValue)
		{
			VM vm = (VM)Entity;
			if (vm == null)
			{
				return;
			}

			//Check that we get a return value corresponding to a current VM.
			foreach (AsyncDataProvider.GetSnapshotListQueryResult result in (IList<AsyncDataProvider.GetSnapshotListQueryResult>)returnValue)
			{
				if (result.VmId != vm.vm_guid)
				{
					return;
				}
			}



			data = new Dictionary<DiskImage, IList<DiskImage>>();
			previewingImages = new Dictionary<Guid, Guid>();

			foreach (AsyncDataProvider.GetSnapshotListQueryResult result in (IList<AsyncDataProvider.GetSnapshotListQueryResult>)returnValue)
			{
				data.Add(result.Disk, result.Snapshots);
				previewingImages.Add(result.Disk.Id, result.PreviewingImage);
			}


			UpdateItems();
			UpdateActionAvailability();
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();

			UpdateActionAvailability();


			List<string> list = new List<string>();
			if (SelectedItem != null && SelectedItem.Apps != null)
			{
				foreach (string item in SelectedItem.Apps.Split(','))
				{
					list.Add(item);
				}
			}

			Apps = list;
		}

		protected override void SelectedItemsChanged()
		{
			base.SelectedItemsChanged();
			UpdateActionAvailability();
		}

		protected override void EntityPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.EntityPropertyChanged(sender, e);

			if (e.PropertyName == "status")
			{
				UpdateData();
			}
		}

		private IDictionary<DiskImage, IList<DiskImage>> data;
		private IDictionary<Guid, Guid> previewingImages;


		public void UpdateItems()
		{
			if (Entity == null)
			{
				Items = null;
				return;
			}

			VM vm = (VM)Entity;
			List<DiskImage> images = new List<DiskImage>();

			foreach (IList<DiskImage> item in data.Values)
			{
				List<DiskImage> temp = new List<DiskImage>();
				temp.AddRange(item);

				if (item.Count > 0)
				{
					temp.RemoveAt(0);
				}

				images.AddRange(temp);
			}

			//Combine snapshots by disk.
			//Items = l
			//    .Where(a => a.lastModified.Date == Date.Date)
			//    .OrderByDescending(a => a.lastModified.TimeOfDay)
			//    .GroupBy(a => a.vm_snapshot_id)
			//    .Select(a =>
			//        new SnapshotModel
			//        {
			//            SnapshotId = a.Key.GetValueOrDefault(),
			//            Date = a.First().lastModified,
			//            IsPreviewed = a.Any(b => previewingImages.Any(c => b.image_guid == c.Value)),
			//            Description = a.Any(b => previewingImages.Any(c => b.image_guid == c.Value))
			//                          ? new EntityModel() { Value = a.First().description + " (Preview Mode)" }
			//                          : new EntityModel() { Value = a.First().description },
			//            ParticipantDisks = a.Select(b => b.internal_drive_mapping)
			//                .Distinct()
			//                .OrderBy(b => b)
			//                .Separate(',', b => b),
			//            Apps = a.First().appList
			//        }
			//    )
			//    .ToList();

			Linq.OrderByDescending(images, new Linq.DiskImageByLastModifiedTimeOfDayComparer());

			Dictionary<nguid, List<DiskImage>> dict = new Dictionary<nguid, List<DiskImage>>();
			foreach (DiskImage a in images)
			{
				if (!dict.ContainsKey(a.vm_snapshot_id))
				{
					dict.Add(a.vm_snapshot_id, new List<DiskImage>());
				}

				List<DiskImage> ls = dict[a.vm_snapshot_id];
				ls.Add(a);
			}


			List<SnapshotModel> items = new List<SnapshotModel>();

			//Add hard coded "Current" snapshot.
			SnapshotModel m = new SnapshotModel();
			m.IsCurrent = true;
			m.ParticipantDisks = DriveNamesAsString(getDriveNames(data.Keys));
			m.Description.Entity = "<" + vm.vm_name + ">";
			items.Add(m);

			foreach (KeyValuePair<nguid, List<DiskImage>> pair in dict)
			{
				nguid key = pair.Key;
				List<DiskImage> value = pair.Value;

				DiskImage firstDiskImage = Linq.FirstOrDefault(value);

				m = new SnapshotModel();
				m.SnapshotId = key.Value;
				m.Date = firstDiskImage.lastModified;
				m.Apps = firstDiskImage.appList;

				//IsPreviewed
				bool isPreviewed = false;
				foreach (DiskImage b in value)
				{
					if (isPreviewed)
						break;

					foreach (KeyValuePair<Guid, Guid> c in previewingImages)
					{
						if (b.Id.Equals(c.Value))
						{
							isPreviewed = true;
							break;
						}
					}
				}
				m.IsPreviewed = isPreviewed;

				//Description
				string description = firstDiskImage.description;
				if (isPreviewed)
				{
					description += " (Preview Mode)";
				}
				m.Description = new EntityModel { Entity = description };

				m.ParticipantDisks = DriveNamesAsString(getDriveNames(value));

				items.Add(m);
			}

			items = Linq.OrderByDescending(items, new Linq.SnapshotModelDateComparer());
			Items = items;
		}

		private string DriveNamesAsString(List<string> names)
		{
			string result = String.Empty;
			foreach (string a in names)
			{
				result += a;
				if (a != names[names.Count - 1])
				{
					result += ", ";
				}
			}

			return result;
		}

		private List<string> getDriveNames(IEnumerable<DiskImage> images)
		{
			//ParticipantDisks
			List<string> list = new List<string>();
			foreach (DiskImage a in images)
			{
				if (!list.Contains(a.internal_drive_mapping))
				{
					list.Add("Disk " + a.internal_drive_mapping);
				}
			}
			list.Sort();
			return list;
		}

		public void UpdateActionAvailability()
		{
			if (Entity is vm_pools)
			{
				return;
			}

			VM vm = (VM)Entity;

			//bool isPreviewing = previewingImages.Any(a => a.Value != Guid.Empty);
			bool isPreviewing = false;
			foreach (KeyValuePair<Guid, Guid> a in previewingImages)
			{
				if (!a.Value.Equals(Guid.Empty))
				{
					isPreviewing = true;
					break;
				}
			}

			bool isVmDown = vm == null || vm.status == VMStatus.Down;
			bool isVmImageLocked = vm == null || vm.status == VMStatus.ImageLocked;

			PreviewCommand.IsExecutionAllowed = !isPreviewing
				&& SelectedItem != null
				&& isVmDown;

			CanSelectSnapshot.Entity = !isPreviewing && !isVmImageLocked;

			NewCommand.IsExecutionAllowed = !isPreviewing
				&& isVmDown;

			CommitCommand.IsExecutionAllowed = isPreviewing
				&& isVmDown;

			UndoCommand.IsExecutionAllowed = isPreviewing
				&& isVmDown;

			RemoveCommand.IsExecutionAllowed = !isPreviewing
				&& SelectedItem != null
				&& isVmDown;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == NewCommand)
			{
				New();
			}
			else if (command == PreviewCommand)
			{
				Preview();
			}
			else if (command == CommitCommand)
			{
				Commit();
			}
			else if (command == UndoCommand)
			{
				Undo();
			}
			else if (command == RemoveCommand)
			{
				Remove();
			}
			else if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
			else if (command.Name == "OnNew")
			{
				OnNew();
			}
		}
	}


	public class GetAllVmSnapshotsExecutor : IFrontendMultipleQueryAsyncCallback
	{
		private readonly Guid vmId;
		private readonly AsyncQuery query;
		private IList<DiskImage> disks;

		public GetAllVmSnapshotsExecutor(Guid vmId, AsyncQuery query)
		{
			this.vmId = vmId;
			this.query = query;
		}

		public void Execute()
		{
			AsyncDataProvider.GetVmDiskList(new AsyncQuery(this,
				(target, returnValue) =>
				{
					GetAllVmSnapshotsExecutor model = (GetAllVmSnapshotsExecutor)target;

					model.PostExecute(returnValue);
				})
				, vmId
			);
		}

		public void PostExecute(object returnValue)
		{
			disks = Linq.Cast<DiskImage>((IEnumerable)(valueObjectEnumerableList)returnValue);

			List<VdcQueryType> queryTypes = new List<VdcQueryType>();
			List<VdcQueryParametersBase> parameters = new List<VdcQueryParametersBase>();

			foreach (DiskImage disk in disks)
			{
				queryTypes.Add(VdcQueryType.GetAllVmSnapshotsByDrive);
				parameters.Add(new GetAllVmSnapshotsByDriveParameters(vmId, disk.internal_drive_mapping));
			}

			Frontend.RunMultipleQueries(queryTypes, parameters, this);
		}

		public void Executed(FrontendMultipleQueryAsyncResult result)
		{
			List<AsyncDataProvider.GetSnapshotListQueryResult> list = new List<AsyncDataProvider.GetSnapshotListQueryResult>();

			for (int i = 0; i < result.ReturnValues.Count; i++)
			{
				GetAllVmSnapshotsByDriveQueryReturnValue returnValue = (GetAllVmSnapshotsByDriveQueryReturnValue)result.ReturnValues[i];

				list.Add(
					new AsyncDataProvider.GetSnapshotListQueryResult(
						returnValue.TryingImage,
						Linq.Cast<DiskImage>((IEnumerable)(valueObjectEnumerableList)returnValue.ReturnValue),
						disks[i])
						{
							VmId = vmId
						}
					);
			}

			query.asyncCallback(query.Model, list);
		}
	}
}
