﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Windows.Controls;
using System.Windows;
using System.Windows.Input;
using System.Globalization;
using System.Collections;
using System.Windows.Controls.Primitives;

namespace UI.WPFClient
{
	[TemplatePart(Type = typeof(Panel), Name = "PART_DaysPanel")]
	public class Calendar : Control
	{
		#region Commands

		public static RoutedCommand NextMonthCommand = new RoutedCommand("NextMonth", typeof(Calendar));
		public static RoutedCommand PreviousMonthCommand = new RoutedCommand("PreviousMonth", typeof(Calendar));

		#endregion

		#region Properties

		public DateTime Date
		{
			get { return (DateTime)GetValue(DateProperty); }
			set { SetValue(DateProperty, value); }
		}
		public static readonly DependencyProperty DateProperty =
			DependencyProperty.Register("Date", typeof(DateTime), typeof(Calendar), new UIPropertyMetadata(DateTime.Now, DateChanged));

		private static void DateChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
		{
			Calendar instance = (Calendar)d;
			instance.DateChanged((DateTime)e.OldValue, (DateTime)e.NewValue);
		}


		public DateTime MaxDate
		{
			get { return (DateTime)GetValue(MaxDateProperty); }
			set { SetValue(MaxDateProperty, value); }
		}
		public static readonly DependencyProperty MaxDateProperty =
			DependencyProperty.Register("MaxDate", typeof(DateTime), typeof(Calendar), new UIPropertyMetadata());


		public DateTime MinDate
		{
			get { return (DateTime)GetValue(MinDateProperty); }
			set { SetValue(MinDateProperty, value); }
		}
		public static readonly DependencyProperty MinDateProperty =
			DependencyProperty.Register("MinDate", typeof(DateTime), typeof(Calendar), new UIPropertyMetadata());


		public DateTime Today
		{
			get { return (DateTime)GetValue(TodayProperty); }
			set { SetValue(TodayProperty, value); }
		}
		public static readonly DependencyProperty TodayProperty =
			DependencyProperty.Register("Today", typeof(DateTime), typeof(Calendar), new UIPropertyMetadata());

        public DateTime PreviewedDay
        {
            get { return (DateTime)GetValue(PreviewedDayProperty); }
            set { SetValue(PreviewedDayProperty, value); }
        }
        public static readonly DependencyProperty PreviewedDayProperty =
            DependencyProperty.Register("PreviewedDay", typeof(DateTime), typeof(Calendar), new UIPropertyMetadata());

		public IEnumerable DaysOfWeek
		{
			get { return (IEnumerable)GetValue(DaysOfWeekProperty); }
			set { SetValue(DaysOfWeekProperty, value); }
		}
		public static readonly DependencyProperty DaysOfWeekProperty =
			DependencyProperty.Register("DaysOfWeek", typeof(IEnumerable), typeof(Calendar), new UIPropertyMetadata(null));


		public Style DayStyle
		{
			get { return (Style)GetValue(DayStyleProperty); }
			set { SetValue(DayStyleProperty, value); }
		}
		public static readonly DependencyProperty DayStyleProperty =
			DependencyProperty.Register("DayStyle", typeof(Style), typeof(Calendar), new UIPropertyMetadata(null));


		public IDictionary<DateTime, object> DayDataItems
		{
			get { return (IDictionary<DateTime, object>)GetValue(DayDataItemsProperty); }
			set { SetValue(DayDataItemsProperty, value); }
		}
		public static readonly DependencyProperty DayDataItemsProperty =
			DependencyProperty.Register("DayDataItems", typeof(IDictionary<DateTime, object>), typeof(Calendar), new UIPropertyMetadata(null, DayDataItemsChanged));

		private static void DayDataItemsChanged(DependencyObject d, DependencyPropertyChangedEventArgs e)
		{
			Calendar instance = (Calendar)d;
			instance.TryInitDays();
	
		}

		#endregion

		private Panel daysPanel;

		private CalendarDay selectedDay;

		public Calendar()
		{
			CommandBindings.Add(new CommandBinding(NextMonthCommand, NextMonthExecuted));
			CommandBindings.Add(new CommandBinding(PreviousMonthCommand, PreviousMonthExecuted));

			DaysOfWeek = CultureInfo.CurrentUICulture.DateTimeFormat.ShortestDayNames;
		}

		private void PreviousMonthExecuted(object sender, ExecutedRoutedEventArgs e)
		{
			int year = Date.Year;
			int month = Date.Month;
			int prevMonth = month == 1 ? 12 : month - 1;
			int prevMonthYear = month == 1 ? year - 1 : year;
			Date = new DateTime(prevMonthYear, prevMonth, 1);
			UpdateSelectedDay();

			e.Handled = true;
		}

		private void NextMonthExecuted(object sender, ExecutedRoutedEventArgs e)
		{
			Date = Date.AddMonths(1);
			UpdateSelectedDay();

			e.Handled = true;
		}

		private void DayClick(object sender, RoutedEventArgs e)
		{
			CalendarDay day = (CalendarDay)e.OriginalSource;
			Date = day.Date;
			UpdateSelectedDay();

			e.Handled = true;
		}

		public override void OnApplyTemplate()
		{
			base.OnApplyTemplate();

			daysPanel = GetTemplateChild("PART_DaysPanel") as Panel;
			TryInitDays();
		}

		private void DateChanged(DateTime oldDate, DateTime newDate)
		{
			if (newDate.Year != oldDate.Year || newDate.Month != oldDate.Month)
			{
				TryInitDays();
			}
			else if (newDate.Year == oldDate.Year && newDate.Month == oldDate.Month && newDate.Day != oldDate.Day)
			{
				UpdateSelectedDay();
			}
		}

		public void TryInitDays()
		{
			if (daysPanel != null && !DesignerProperties.GetIsInDesignMode(this))
			{
				daysPanel.Children.Clear();

				var dtf = CultureInfo.CurrentUICulture.DateTimeFormat;
				int year = Date.Year;
				int month = Date.Month;
				int prevMonth = month == 1 ? 12 : month - 1;
				int prevMonthYear = month == 1 ? year - 1 : year;
				//int daysInMonth = DateTime.DaysInMonth(year, month);
				int daysInPrevMonth = DateTime.DaysInMonth(prevMonthYear, prevMonth);
				DayOfWeek firstDayOfWeek = dtf.FirstDayOfWeek;
				DayOfWeek firstDayOfMonth = new DateTime(year, month, 1).DayOfWeek;

				//We have 42 days to display.

				int r = 7;
				if (firstDayOfMonth != firstDayOfWeek)
				{
					r = firstDayOfMonth - firstDayOfWeek;
				}

				DateTime startDate = new DateTime(prevMonthYear, prevMonth, daysInPrevMonth - r + 1);
				for (int i = 0; i < 42; i++)
				{
					DateTime useDate = startDate.AddDays(i);

					//Find day data item.
					object dataItem = null;
					if (DayDataItems != null && DayDataItems.ContainsKey(useDate.Date))
					{
						dataItem = DayDataItems[useDate];
					}

					var day =
						new CalendarDay
						{
							Date = useDate,
							Style = DayStyle,
							IsBelongsToSelectedMonth = useDate.Month == Date.Month,
							IsToday = useDate.Date == Today.Date,
                            IsPreviewed = useDate.Date == PreviewedDay.Date,
							DataContext = dataItem
						};
					day.Click += DayClick;

					daysPanel.Children.Add(day);
				}

				UpdateSelectedDay();
			}
		}

		private void UpdateSelectedDay()
		{
			//Unselect previously selected day.
			if (selectedDay != null)
			{
				Selector.SetIsSelected(selectedDay, false);
			}

			//Find the right calendar day.
			CalendarDay day = daysPanel.Children
				.Cast<CalendarDay>()
				.First(a => a.Date.Date == Date.Date);

			Selector.SetIsSelected(day, true);
			selectedDay = day;
		}

		public CalendarDay GetDayByDate(DateTime date)
		{
			return null;
		}
	}


	public class CalendarDay : Button
	{
		#region Properties

		public DateTime Date
		{
			get { return (DateTime)GetValue(DateProperty); }
			set { SetValue(DateProperty, value); }
		}
		public static readonly DependencyProperty DateProperty =
			DependencyProperty.Register("Date", typeof(DateTime), typeof(CalendarDay), new UIPropertyMetadata());


		public bool IsToday
		{
			get { return (bool)GetValue(IsTodayProperty); }
			set { SetValue(IsTodayProperty, value); }
		}
		public static readonly DependencyProperty IsTodayProperty =
			DependencyProperty.Register("IsToday", typeof(bool), typeof(CalendarDay), new UIPropertyMetadata(false));

        public bool IsPreviewed
        {
            get { return (bool)GetValue(IsPreviewedProperty); }
            set { SetValue(IsPreviewedProperty, value); }
        }
        public static readonly DependencyProperty IsPreviewedProperty =
            DependencyProperty.Register("IsPreviewed", typeof(bool), typeof(CalendarDay), new UIPropertyMetadata(false));

        public bool IsBelongsToSelectedMonth
		{
			get { return (bool)GetValue(IsBelongsToSelectedMonthProperty); }
			set { SetValue(IsBelongsToSelectedMonthProperty, value); }
		}
		public static readonly DependencyProperty IsBelongsToSelectedMonthProperty =
			DependencyProperty.Register("IsBelongsToSelectedMonth", typeof(bool), typeof(CalendarDay), new UIPropertyMetadata(false));

		#endregion
	}
}
