﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Configuration;
using System.IO;
using System.Linq;
using System.ServiceModel.Configuration;
using System.ServiceModel.Security;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Windows;
using System.Windows.Documents;
using org.ovirt.engine.ui.uicompat;
using Microsoft.Practices.Unity;
using VdcFrontend;
using org.ovirt.engine.ui.uicommon;
using VdcCommon.BusinessEntities;
using org.ovirt.engine.ui.uicommon.models;
using org.ovirt.engine.ui.uicommon.models.common;
using org.ovirt.engine.ui.uicommon.models.vms;
using System.Windows.Interop;
using System.Security.Cryptography.X509Certificates;
using System.Net;
using System.Net.Security;
using VdcUtils;

namespace UI.WPFClient
{
	public sealed class GeneralFailureDocumentBuilder : IDocumentBuilder
	{
		private readonly IEnumerable<string> items;

		public GeneralFailureDocumentBuilder(IEnumerable<string> items)
		{
			this.items = items;
		}

		public void BuildDocument(FlowDocument document)
		{
			items.Select(a => new Paragraph(new Run(a)))
				.ToList()
				.ForEach(a => document.Blocks.Add(a));
		}
	}

	public sealed class CanDoActionDocumentBuilder : IDocumentBuilder
	{
		private readonly IEnumerable<List<string>> _itemsWithDescription;

		/// <summary>
		/// CanDoActionDocumentBuilder constructor: Gets a list of string sub-lists.
		/// Each sub-list represent the CanDoAction failure info for a single entity.
		/// The first string in each sub-list is the description of the VdcReturnValueBase,
		/// normally containing the name of the relevant entity on which the CanDoAction failed.
		/// NOTE: It might be empty!
		/// The rest of the strings in each sub-list are the CanDoAction failure messages.
		/// </summary>
		/// <param name="itemsWithDescription">the list of description + messages sub-lists.</param>
		public CanDoActionDocumentBuilder(IEnumerable<List<string>> itemsWithDescription)
		{
			_itemsWithDescription = itemsWithDescription;
		}

		public void BuildDocument(FlowDocument document)
		{
			foreach (List<string> entityItems in _itemsWithDescription)
			{
				if (entityItems != null && entityItems.Count > 1)
				{
					// We check that entityItems have at least two items, since if 
					// there is only one item, it is the entity name and there is no point
					// in showing "entityName:" without messages under it.

					// Title (description):
					Paragraph title =
						new Paragraph(
							new Run(
								string.Format(
									"{0}{1}",
									entityItems[0],
									string.IsNullOrEmpty(entityItems[0].Trim()) ? string.Empty : ":")));
					title.Style = (Style)(document.FindResource("CanDoActionTitleParagraphStyle"));
					document.Blocks.Add(title);

					// Items (messages):
					IEnumerable<string> canDoActionMessages = entityItems.Skip(1);
					List itemsList = new List();
					foreach (string item in canDoActionMessages)
					{
						ListItem listItem = new ListItem(new Paragraph(new Run(item)));
						listItem.Style = (Style)(document.FindResource("CanDoActionListItemStyle"));
						itemsList.ListItems.Add(listItem);
					}
					itemsList.Style = (Style)(document.FindResource("CanDoActionItemsListStyle"));
					document.Blocks.Add(itemsList);
				}
			}
		}
	}

	public sealed class ConnectionClosedDocumentBuilder : IDocumentBuilder
	{
		private readonly Exception exception;

		public ConnectionClosedDocumentBuilder(Exception exception)
		{
			this.exception = exception;
		}

		public void BuildDocument(FlowDocument document)
		{
			document.Blocks.Add(new Paragraph(new Run("Connection Lost, probably due to session time-out. Please login again.")));

			//exception.Flatten().Select(a => new Paragraph(new Run(a.Message)))
			//    .ToList()
			//    .ForEach(a => document.Blocks.Add(a));
			foreach (Exception a in org.ovirt.engine.ui.uicommon.Extensions.Flatten(exception))
			{
				Paragraph p = new Paragraph(new Run(a.Message));
				document.Blocks.Add(p);
			}
		}
	}


	public partial class Shell
	{
		#region Properties

		public ShellModel Model
		{
			get { return (ShellModel)DataContext; }
			set { DataContext = value; }
		}


		public ActiveXContainer ActiveXContainer
		{
			get { return (ActiveXContainer)GetValue(ActiveXContainerProperty); }
			set { SetValue(ActiveXContainerProperty, value); }
		}
		public static readonly DependencyProperty ActiveXContainerProperty =
			DependencyProperty.Register("ActiveXContainer", typeof(ActiveXContainer), typeof(Shell), new UIPropertyMetadata(null));

		#endregion

		private readonly IUnityContainer container;
		private readonly WindowManager windowManager;
		private readonly Configurator configurator;
		private readonly ILogger logger;


		public Shell()
		{
			InitializeComponent();


			container = new UnityContainer();
			TypeResolver.Initialize(new UnityTypeResolver(container));

			container.RegisterInstance(this);

			configurator = new WPFConfigurator();
			container.RegisterInstance(configurator);

			logger =
				new Log4NetLogger
				{
					LogLevel = configurator.LogLevel
				};
			container.RegisterInstance(logger);


			container.RegisterType<AboutModel, WPFAboutModel>();

			ActiveXContainer = new ActiveXContainer();
			container.RegisterInstance(ActiveXContainer);

			container.RegisterType<ISpice, Spice>();
			container.RegisterType<IVnc, Vnc>();
			container.RegisterType<IRdp, Rdp>();

			container.RegisterType<ITimer, WPFTimer>();


			//Update client configuration at this point.
			ModifyServiceModelConfigSection();


			//Resolve window manager instance.
			windowManager = container.Resolve<WindowManager>();
			windowManager.AddDragSurface(DragSurface);
			container.RegisterInstance(windowManager);


			Model = new ShellModel(SynchronizationContext.Current);
			Model.PropertyChanged += Model_PropertyChanged;
			Model.NewMessage += Model_NewMessage;
			Frontend.InitEventsHandler(Model);

			DocumentationPathTranslator.Init(App.RelativePath + ConfigurationManager.AppSettings["DocumentationPathFileName"]);
		}

		private void ModifyServiceModelConfigSection()
		{
			Configuration cfg = ConfigurationManager.OpenExeConfiguration(ConfigurationUserLevel.None);
			ServiceModelSectionGroup group = (ServiceModelSectionGroup)cfg.GetSectionGroup("system.serviceModel");

			foreach (ChannelEndpointElement endpoint in group.Client.Endpoints)
			{
				string address = endpoint.Address.ToString();
				if (endpoint.Address.IsDefaultPort)
				{
					address = address.Replace("localhost", String.Format("{0}:{1}", App.Host, configurator.BackendPort));
				}

				//Update port segment anyway.
				address = Regex.Replace(address, @"(?<a>\w+?://.+?):(?<b>\d+?)/(?<c>.+)", String.Format(@"${{a}}:{0}/${{c}}", configurator.BackendPort));

				endpoint.Address = new Uri(address);
			}

			// "Auto" means to validate server certificate only if we browse with https:
			if ((configurator.ValidateServerCertificate == ValidateServerCertificateEnum.AUTO &&
				BrowserInteropHelper.Source.Scheme != "https") ||
				configurator.ValidateServerCertificate == ValidateServerCertificateEnum.FALSE)
			{
				// auto + !https or "don't validate server certicicate" -> override validation method to return true:
				ServicePointManager.ServerCertificateValidationCallback = new RemoteCertificateValidationCallback(OnValidationCallback);
			}

			cfg.Save();
		}

		public static bool OnValidationCallback(object sender, X509Certificate cert, X509Chain chain, SslPolicyErrors errors)
		{
			return true;
		}

		private void Model_PropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			if (e.PropertyName == "Entity")
			{
				//Entity of the model represents all the content of the shell.
				//Each time it changed close all dialogs.
				windowManager.CloseAll();
			}
		}

		private MessageModel exclusiveModel;

		void Model_NewMessage(object sender, ModelEventArgs<MessageModel> e)
		{
			var model = e.Model;

			bool show = !model.IsExclusive || (model.IsExclusive && exclusiveModel == null);

			//If we got an exclusive message, show only it.
			if (model.IsExclusive && exclusiveModel == null)
			{
				windowManager.CloseAll();
				exclusiveModel = model;
			}

			if (show)
			{
				var view = container.Resolve<MessageView>();
				view.Model = model;
				windowManager.AddWindow(view);

				model.Closed +=
					(s, args) =>
					{
						windowManager.RemoveWindow(view);
						exclusiveModel = null;
					};
			}
		}
	}
}
