using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.validation;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using VdcCommon.VdcQueries;
using System.Collections;
using org.ovirt.engine.ui.uicompat;

namespace org.ovirt.engine.ui.uicommon.models.vms
{
	public class ImportVmModel : ListWithDetailsModel
	{
		#region Properties

		public storage_domain_static SourceStorage { get; set; }
		public storage_pool StoragePool { get; set; }

		public ListModel DestinationStorage { get; private set; }
		public ListModel Cluster { get; private set; }
		public ListModel SystemDiskFormat { get; private set; }
		public ListModel DataDiskFormat { get; private set; }
		public EntityModel CollapseSnapshots { get; private set; }

		private string nameAndDescription;
		public string NameAndDescription
		{
			get { return nameAndDescription; }
			set
			{
				if (nameAndDescription != value)
				{
					nameAndDescription = value;
					OnPropertyChanged(new PropertyChangedEventArgs("NameAndDescription"));
				}
			}
		}

		private IList<VM> problematicItems;
		public IList<VM> ProblematicItems
		{
			get { return problematicItems; }
			set
			{
				if (problematicItems != value)
				{
					problematicItems = value;
					OnPropertyChanged(new PropertyChangedEventArgs("ProblematicItems"));
				}
			}
		}

		private bool hasCollapseSnapshotsWarning;
		public bool HasCollapseSnapshotsWarning
		{
			get { return hasCollapseSnapshotsWarning; }
			set
			{
				if (hasCollapseSnapshotsWarning != value)
				{
					hasCollapseSnapshotsWarning = value;
					OnPropertyChanged(new PropertyChangedEventArgs("HasCollapseSnapshotsWarning"));
				}
			}
		}

		#endregion

		public ImportVmModel()
		{
			CollapseSnapshots = new EntityModel { Entity = false };
			CollapseSnapshots.EntityChangedEvent.addListener(this);
			DestinationStorage = new ListModel();
			DestinationStorage.SelectedItemChangedEvent.addListener(this);
			Cluster = new ListModel();
			SystemDiskFormat = new ListModel();
			DataDiskFormat = new ListModel();
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(EntityChangedEventDefinition) && sender == CollapseSnapshots)
			{
				CollapseSnapshots_EntityChanged();
			}
			else if (ev.Equals(SelectedItemChangedEvent) && sender == DestinationStorage)
			{
				DestinationStorage_SelectedItemChanged();
			}
		}

		private void DestinationStorage_SelectedItemChanged()
		{
			UpdateImportWarnings();
		}

		protected override void ItemsChanged()
		{
			base.ItemsChanged();
			UpdateImportWarnings();
		}

		private void UpdateImportWarnings()
		{
			//Clear problematic state.
			ProblematicItems = null;

			if (Items == null)
			{
				return;
			}


			storage_domains destinationStorage = (storage_domains)DestinationStorage.SelectedItem;

			//Determine which items are problematic.
			List<VM> problematicItems = new List<VM>();

			foreach (object item in Items)
			{
				VM vm = (VM)item;

				if (vm.DiskMap != null)
				{
					foreach (KeyValuePair<string, DiskImage> pair in vm.DiskMap)
					{
						DiskImage disk = pair.Value;

						if (disk.volume_type == VolumeType.Sparse
							&& disk.volume_format == VolumeFormat.RAW
							&& destinationStorage != null
							&& (destinationStorage.storage_type == StorageType.ISCSI || destinationStorage.storage_type == StorageType.FCP))
						{
							problematicItems.Add(vm);
						}
					}
				}
			}

			//Decide what to do with the CollapseSnapshots option.
			if (problematicItems.Count > 0)
		{
				if (problematicItems.Count == Linq.Count(Items))
				{
					//All items are problematic.
					CollapseSnapshots.IsChangable = false;
					CollapseSnapshots.Entity = true;
					CollapseSnapshots.Message = "Note that all snapshots will be collapsed due to different storage types";
					HasCollapseSnapshotsWarning = true;
				}
				else
				{
					//Some items are problematic.
					CollapseSnapshots.Message = "Use a separate import operation for the marked VMs or\nApply \"Collapse Snapshots\" for all VMs";
					HasCollapseSnapshotsWarning = !(bool)CollapseSnapshots.Entity;
					ProblematicItems = (bool)CollapseSnapshots.Entity ? null : problematicItems;
				}
			}
			else
			{
				//No problematic items.
				CollapseSnapshots.IsChangable = true;
				CollapseSnapshots.Message = null;
				HasCollapseSnapshotsWarning = false;
			}
		}

		public void CollapseSnapshots_EntityChanged()
		{
			if (Items == null)
				return;

			storage_domains selectedDestinationStorage = null;
			bool sameSelectedDestinationStorage = false;
			if (DestinationStorage.SelectedItem != null)
			{
				selectedDestinationStorage = (storage_domains)DestinationStorage.SelectedItem;
			}
			List<storage_domains> destStorages = new List<storage_domains>();
			Dictionary<Guid, List<storage_domains>> templateGuidStorageDomainDic =
				new Dictionary<Guid, List<storage_domains>>();
			foreach (object item in Items)
			{
				VM vm = (VM)item;
				Guid guid = vm.vmt_guid;
				if (templateGuidStorageDomainDic.ContainsKey(guid))
					continue;
				if (guid.Equals(Guid.Empty))
				{
					templateGuidStorageDomainDic.Add(guid, null);
				}
				else
				{
					templateGuidStorageDomainDic.Add(guid, DataProvider.GetStorageDomainListByTemplate(guid));
				}
			}
			foreach (storage_domains domain in DataProvider.GetDataDomainsListByDomain(this.SourceStorage.Id))
			{
				bool addStorage = false;
				if ((domain.storage_domain_type == StorageDomainType.Data ||
					 domain.storage_domain_type == StorageDomainType.Master)
					&& domain.status.HasValue && domain.status.Value == StorageDomainStatus.Active)
				{
					foreach (KeyValuePair<Guid, List<storage_domains>> keyValuePair in templateGuidStorageDomainDic)
					{
						if (Guid.Empty.Equals(keyValuePair.Key))
						{
							addStorage = true;
						}
						else
						{
							addStorage = false;
							foreach (storage_domains storageDomain in keyValuePair.Value)
							{
								if (storageDomain.id.Equals(domain.id) || (bool)CollapseSnapshots.Entity)
								{
									addStorage = true;
									break;
								}
							}
						}
						if (addStorage == false)
						{
							break;
						}
					}
				}
				if (addStorage)
				{
					destStorages.Add(domain);
					if (sameSelectedDestinationStorage == false && domain.Equals(selectedDestinationStorage))
					{
						sameSelectedDestinationStorage = true;
						selectedDestinationStorage = domain;
					}
				}
			}
			DestinationStorage.Items = destStorages;
			if (sameSelectedDestinationStorage)
			{
				DestinationStorage.SelectedItem = selectedDestinationStorage;
			}
			else
			{
				DestinationStorage.SelectedItem = Linq.FirstOrDefault(destStorages);
			}

			if (DetailModels != null && ActiveDetailModel is VmImportDiskListModel)
			{
				VmImportDiskListModel detailModel = (VmImportDiskListModel)ActiveDetailModel;
				detailModel.CollapseSnapshots = (bool)CollapseSnapshots.Entity;
			}


			UpdateImportWarnings();
		}

		protected override void ActiveDetailModelChanged()
		{
			base.ActiveDetailModelChanged();
			CollapseSnapshots_EntityChanged();
		}

		protected override void InitDetailModels()
		{
			base.InitDetailModels();

			VmImportDiskListModel importDiskListModel = new VmImportDiskListModel();

			ObservableCollection<EntityModel> list = new ObservableCollection<EntityModel>();
			list.Add(new VmGeneralModel());
			list.Add(new VmImportInterfaceListModel());
			list.Add(importDiskListModel);
			list.Add(new VmAppListModel());
			DetailModels = list;

			importDiskListModel.SelectedVMsCount = ((IList)Items).Count;
		}

		internal bool Validate()
		{
			DestinationStorage.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });
			Cluster.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });

			return DestinationStorage.IsValid
				   && Cluster.IsValid;
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();

			if (SelectedItem != null)
			{
				VM vm = (VM)SelectedItem;
				NameAndDescription = string.Format("{0}{1}", vm.vm_name, !String.IsNullOrEmpty(vm.vm_description) ? " [" + vm.vm_description + "]" : String.Empty);
			}
			else
			{
				NameAndDescription = string.Empty;
			}
		}
	}
}
