using System;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Text;
using org.ovirt.engine.ui.uicommon.dataprovider;
using org.ovirt.engine.ui.uicompat;
using org.ovirt.engine.ui.uicommon.validation;
using VdcCommon.Users;
using VdcCommon.BusinessEntities;
using System.Reflection;
using VdcFrontend;
using org.ovirt.engine.ui.uicommon.models.common;

namespace org.ovirt.engine.ui.uicommon.models
{
	public class LoginModel : Model, ITaskTarget
	{
		#region Const

		public const string BeginLoginStage = "BeginTest";
		public const string EndLoginStage = "EndTest";

		#endregion Const

		#region Events

		public static EventDefinition LoggedInEventDefinition;
		public Event LoggedInEvent { get; private set; }

		public static EventDefinition LoginFailedEventDefinition;
		public Event LoginFailedEvent { get; private set; }

		#endregion

		#region Commands

		public UICommand LoginCommand { get; set; }
		public UICommand AboutCommand { get; private set; }


		#endregion

		#region Properties

		private Model window;
		public Model Window
		{
			get { return window; }
			set
			{
				if (window != value)
				{
					window = value;
					OnPropertyChanged(new PropertyChangedEventArgs("Window"));
				}
			}
		}

		public ListModel Domain { get; private set; }
		public EntityModel UserName { get; private set; }
		public EntityModel Password { get; private set; }

		private bool isConnecting;
		public bool IsConnecting
		{
			get { return isConnecting; }
			set
			{
				if (isConnecting != value)
				{
					isConnecting = value;
					OnPropertyChanged(new PropertyChangedEventArgs("IsConnecting"));
				}
			}
		}

		#endregion


		static LoginModel()
		{
			LoggedInEventDefinition = new EventDefinition("LoggedIn", typeof(LoginModel));
			LoginFailedEventDefinition = new EventDefinition("LoginFailed", typeof(LoginModel));
		}

		public LoginModel()
		{
			LoggedInEvent = new Event(LoggedInEventDefinition);
			LoginFailedEvent = new Event(LoginFailedEventDefinition);

			LoginCommand =
				new UICommand("Login", this)
				{
					IsExecutionAllowed = false
				};

			AboutCommand =
				new UICommand("About", this)
				{
					IsExecutionAllowed = false
				};

			Domain = new ListModel();
			Domain.IsChangable = false;
			UserName = new EntityModel();
			UserName.IsChangable = false;
			UserName.EntityChangedEvent.addListener(this);
			Password = new EntityModel();
			Password.IsChangable = false;

			IsConnecting = true;

			AsyncQuery _asyncQuery = new AsyncQuery();
			_asyncQuery.HandleFailure = true;
			_asyncQuery.Model = this;
			/*START_DELEGATE*/_asyncQuery.asyncCallback = delegate(Object model, Object ReturnValue)
			{
				IsConnecting = false;

				LoginModel loginModel = (LoginModel)model;
				if(ReturnValue == null)
				{
					loginModel.Message = "Could not connect to RHEV Manager Service, please try to refresh the page. If the problem persists contact your System Administrator.";
					return;
				}
				AsyncQuery _asyncQuery1 = new AsyncQuery();
				_asyncQuery1.Model = loginModel;
				/*START_DELEGATE*/_asyncQuery1.asyncCallback = delegate(Object model1, Object ReturnValue1)
				{
					LoginModel loginModel1 = (LoginModel)model1;

					loginModel1.LoginCommand.IsExecutionAllowed = true;
					loginModel1.AboutCommand.IsExecutionAllowed = true;
					loginModel1.UserName.IsChangable = true;
					loginModel1.Password.IsChangable = true;
					loginModel1.Domain.IsChangable = true;

					IList<string> domains = (IList<string>)ReturnValue1;
					loginModel1.Domain.Items = domains;
					loginModel1.Domain.SelectedItem = Linq.FirstOrDefault(domains);
				};//END_DELEGATE
				AsyncDataProvider.GetDomainListViaPublic(_asyncQuery1, false);
			};//END_DELEGATE
			AsyncDataProvider.IsBackendAvailable(_asyncQuery);
		}

		public override void eventRaised(Event ev, object sender, EventArgs args)
		{
			base.eventRaised(ev, sender, args);

			if (ev.Equals(EntityModel.EntityChangedEventDefinition) && sender == UserName)
			{
				UserName_EntityChanged();
			}
		}

		private void UserName_EntityChanged()
		{
			Domain.IsChangable = GetDomainAvailability();
		}

		private bool GetDomainAvailability()
		{
			//Check whether the user name contains domain part.
			bool hasDomain = GetUserNameParts((string)UserName.Entity)[1] != null;

			return !hasDomain;
		}

		private string[] GetUserNameParts(string value)
		{
			if (!String.IsNullOrEmpty(value))
			{
				int index = value.IndexOf('@');

				//Always return array of two elements representing user name and domain.)
				return
					new string[]
					{
						index > -1 ? value.Remove(index) : value,
						index > -1 ? value.Substring(index + 1) : null
					};
			}

			return new string[] { String.Empty, null };
		}

		public virtual void Login()
		{
			if (!Validate())
			{
				return;
			}

			UserName.IsChangable = false;
			Password.IsChangable = false;
			Domain.IsChangable = false;
			LoginCommand.IsExecutionAllowed = false;

			//Clear config cache on login (to make sure we don't use old config in a new session)
			DataProvider.ClearConfigCache();

			string fullUserName = (string)UserName.Entity;
			string[] parts = GetUserNameParts(fullUserName);

			Task.Create(this,
				new List<object>
				{
					BeginLoginStage,
					fullUserName,
					Password.Entity,
					parts[1],
				})
				.Run();
		}

		protected virtual bool Validate()
		{
			UserName.ValidateEntity(new IValidation[] { new NotEmptyValidation() });
			Password.ValidateEntity(new IValidation[] { new NotEmptyValidation() });
			Domain.ValidateSelectedItem(new IValidation[] { new NotEmptyValidation() });

			return UserName.IsValid
				   && Password.IsValid
				   && Domain.IsValid;
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command == LoginCommand)
			{
				Login();
			}
			else if (command == AboutCommand)
			{
				About();
			}
			else if (command.Name == "Cancel")
			{
				Cancel();
			}
		}

		public void About()
		{
			AboutModel model = new AboutModel();
			Window = model;
			model.Title = "About RHEV Manager";
			model.HashName = "about_rhev_manager";
			model.ShowOnlyVersion = true;

			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Close",
					IsDefault = true,
					IsCancel = true
				});
		}

		public void Cancel()
		{
			Window = null;
		}

		public void run(TaskContext context)
		{
			List<object> state = (List<object>)context.State;
			string stage = (string)state[0];

			switch (stage)
			{
				case BeginLoginStage:
					{
						string fullUserName = (string)state[1];
						string password = (string)state[2];
						string domain = (string)state[3];

						VdcUser user = Frontend.Login(fullUserName, password, String.IsNullOrEmpty(domain) ? (string)Domain.SelectedItem : domain);
						
						context.InvokeUIThread(this,
							new List<object>
							{
								EndLoginStage,
								user
							});
					}
					break;

				case EndLoginStage:
					{
						VdcUser user = (VdcUser)state[1];

						if (user == null)
						{
							Password.Entity = String.Empty;
							LoginFailedEvent.raise(this, EventArgs.Empty);
						}
						else
						{
							LoggedInEvent.raise(this, EventArgs.Empty);
						}
					}
					break;
			}
			

			UserName.IsChangable = true;
			Password.IsChangable = true;
			Domain.IsChangable = GetDomainAvailability();
			LoginCommand.IsExecutionAllowed = true;
		}
	}
}
