using System;
using System.Collections.Generic;
using System.Text;
using org.ovirt.engine.ui.uicommon.models.vms;
using org.ovirt.engine.ui.uicompat;
using VdcCommon.VdcQueries;
using VdcCommon.BusinessEntities;
using VdcFrontend;
using System.Threading;
using System.Windows.Threading;
using System.Diagnostics;
using System.Collections;
using System.ComponentModel;

namespace org.ovirt.engine.ui.uicommon.models.storage
{
	public class VmBackupModel : ManageBackupModel, ITaskTarget
	{
		#region Properties

		public new VM SelectedItem
		{
			get { return (VM)base.SelectedItem; }
			set { base.SelectedItem = value; }
		}

		public VmAppListModel AppListModel { get; private set; }

		#endregion

		public VmBackupModel()
		{
			Title = "VM Import";

			AppListModel = new VmAppListModel();
		}

		protected override void OnSelectedItemChanged()
		{
			base.OnSelectedItemChanged();
			AppListModel.Entity = SelectedItem;
		}

		protected override void Remove()
		{
			base.Remove();

			if (Window != null)
			{
				return;
			}

			ConfirmationModel model = new ConfirmationModel();
			Window = model;
			model.Title = "Remove Backed up VM(s)";
			model.HashName = "remove_backed_up_vm";
			model.Message = "VM(s)";
			
			List<string> items = new List<string>();
			foreach (object item in SelectedItems)
			{
				VM vm = (VM)item;
				items.Add(vm.vm_name);
			}
			model.Items = items;

			model.Note = "Note: The deleted items might still appear on the sub-tab, since the remove operation might be long. Use the Refresh button, to get the updated status.";

			model.Commands.Add(
				new UICommand("OnRemove", this)
				{
					Title = "OK",
					IsDefault = true
				});
			model.Commands.Add(
				new UICommand("Cancel", this)
				{
					Title = "Cancel",
					IsCancel = true
				});
		}

		public void OnRemove()
		{
			ConfirmationModel model = (ConfirmationModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			storage_pool pool = DataProvider.GetFirstStoragePoolByStorageDomain(Entity.id);

			List<VdcActionParametersBase> list = new List<VdcActionParametersBase>();
			foreach (object item in SelectedItems)
			{
				VM vm = (VM)item;
				list.Add(new RemoveVmFromImportExportParamenters(vm, Entity.id, pool.Id));
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.RemoveVmFromImportExport, list,
				result =>
				{
					ConfirmationModel localModel = (ConfirmationModel)result.State;

					localModel.StopProgress();
					Cancel();
					OnEntityChanged();
				},
				model
			);
		}

		protected override void Restore()
		{
			base.Restore();

			if (Window != null)
			{
				return;
			}

			ImportVmModel model = new ImportVmModel();
			Window = model;
			model.Title = "Import Virtual Machine(s)";
			model.HashName = "import_virtual_machine";
			List<VDSGroup> clusters = DataProvider.GetClusterListByStorageDomain(Entity.id);

			model.Cluster.Items = clusters;
			model.Cluster.SelectedItem = Linq.FirstOrDefault(clusters);

			model.SourceStorage = Entity.StorageStaticData;
			model.StoragePool = DataProvider.GetFirstStoragePoolByStorageDomain(Entity.StorageStaticData.Id);

			model.Items = SelectedItems;
			model.CollapseSnapshots_EntityChanged();

			if (((IList)model.DestinationStorage.Items).Count == 0)
			{
				model.DestinationStorage.IsChangable = false;
				model.DestinationStorage.ChangeProhibitionReasons.Add("Cannot import Virtual Machine.");

				model.Message = @"There is no target Data Storage Domain that contains all the Templates which the selected VMs are based on.
Suggested solutions:
1. Preserving Template-Based structure:
   a. Make sure the relevant Templates (on which the VMs are based) exist on the relevant Data-Center.
   b. Import the VMs one by one
2. Using non-Template-Based structure (less optimal storage-wise):
   a. Export the VMs again using the Collapse Snapshots option
   b. Import the VMs.";


				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Close",
						IsDefault = true,
						IsCancel = true
					});
			}
			else
			{
				model.Commands.Add(
					new UICommand("OnRestore", this)
					{
						Title = "OK",
						IsDefault = true
					});
				model.Commands.Add(
					new UICommand("Cancel", this)
					{
						Title = "Cancel",
						IsCancel = true
					});
			}
		}

		public void OnRestore()
		{
			ImportVmModel model = (ImportVmModel)Window;

			if (model.Progress != null)
			{
				return;
			}

			if (!model.Validate())
			{
				return;
			}

			List<VdcActionParametersBase> parameters = new List<VdcActionParametersBase>();

			foreach (object item in SelectedItems)
			{
				VM vm = (VM)item;

				bool collapseSnapshots = (bool)model.CollapseSnapshots.Entity;

				//If no collapse snapshots specified drop problematic VMs.
				if (!collapseSnapshots && model.ProblematicItems != null)
				{
					if (Linq.FirstOrDefault(model.ProblematicItems, new Linq.VmPredicate(vm.vm_guid)) != null)
				{
						continue;
					}
				}

				VDSGroup cluster = (VDSGroup)model.Cluster.SelectedItem;
				storage_domains destinationStorage = (storage_domains)model.DestinationStorage.SelectedItem;

				Dictionary<string, DiskImageBase> disks = new Dictionary<string, DiskImageBase>();
				foreach (KeyValuePair<string, DiskImage> a in vm.DiskMap)
				{
					DiskImage disk = a.Value;
					//Assign a right volume format for each disk.
					disk.volume_format = DataProvider.GetDiskVolumeFormat(disk.volume_type, destinationStorage.storage_type);

					disks.Add(a.Key, a.Value);
				}

				ImportVmParameters parameter =
					new ImportVmParameters(
						vm,
						model.SourceStorage.Id,
						destinationStorage.id,
						model.StoragePool.Id,
						cluster.ID
					)
					{
						ForceOverride = true,
						CopyCollapse = collapseSnapshots,
						DiskInfoList = disks
					};

				parameters.Add(parameter);
			}


			model.StartProgress(null);

			Frontend.RunMultipleAction(VdcActionType.ImportVm, parameters,
				result =>
					{
						VmBackupModel vmBackupModel = (VmBackupModel)result.State;
						vmBackupModel.Window.StopProgress();
						vmBackupModel.Cancel();
						List<VdcReturnValueBase> retVals = (List<VdcReturnValueBase>)result.ReturnValue;
						if (retVals != null && vmBackupModel.SelectedItems.Count > retVals.Count)
						{
							ConfirmationModel confirmModel = new ConfirmationModel();
							vmBackupModel.ConfirmWindow = confirmModel;
							confirmModel.Title = "Import Virtual Machine(s)";
							confirmModel.HashName = "import_virtual_machine";

							string importedVms = String.Empty;
							int i = 0;
							foreach (object item in vmBackupModel.SelectedItems)
							{
								VM vm = (VM)item;
								if (Linq.FindVdcReturnValueByDescription(retVals, vm.vm_name) == null)
								{
									importedVms += vm.vm_name + (++i != vmBackupModel.SelectedItems.Count ? ", " : "");
								}
							}
							importedVms.Trim().TrimEnd(',');

							confirmModel.Message = string.Format("Import process has begun for VM(s): {0}.\nYou can check import status in the 'Events' tab of the specific destination storage domain, or in the main 'Events' tab", importedVms);

							confirmModel.Commands.Add(
								new UICommand("CancelConfirm", vmBackupModel)
								{
									Title = "Close",
									IsDefault = true,
									IsCancel = true
								});
						}
				}, this
				);
		}

		protected override void EntityPropertyChanged(object sender, PropertyChangedEventArgs e)
		{
			base.EntityPropertyChanged(sender, e);

			if (e.PropertyName == "storage_domain_shared_status")
			{
				SearchCommand.Execute();
			}
		}

		protected override void SyncSearch()
		{
			base.SyncSearch();

			if (Entity == null || Entity.storage_domain_type != StorageDomainType.ImportExport || Entity.storage_domain_shared_status != StorageDomainSharedStatus.Active)
			{
				Items = null;
				IsRefreshing = false;
			}
			else
			{
				IsRefreshing = true;
				Task.Create(this, 1).Run();
			}
		}

		protected override void AsyncSearch()
		{
			base.AsyncSearch();
			SyncSearch();
		}

		public override void ExecuteCommand(UICommand command)
		{
			base.ExecuteCommand(command);

			if (command.Name == "OnRemove")
			{
				OnRemove();
			}
			else if (command.Name == "OnRestore")
			{
				OnRestore();
			}
		}

		private VdcQueryReturnValue returnValue;
		private Guid storageDomainId = Guid.Empty;

		public void run(TaskContext context)
		{
			if (Entity != null)
			{

				switch ((int)context.State)
				{
					case 1:
						storageDomainId = Entity.id;
						storage_pool pool = DataProvider.GetFirstStoragePoolByStorageDomain(storageDomainId);

						returnValue = Frontend.RunQuery(VdcQueryType.GetVmsFromExportDomain,
							new GetAllFromExportDomainQueryParamenters(pool.Id, storageDomainId)
							{
								GetAll = true
							});

						context.InvokeUIThread(this, 2);
						break;

					case 2:
						if (storageDomainId.Equals(Entity.id))
						{
							if (returnValue != null && returnValue.Succeeded)
							{
								Items = (List<VM>)returnValue.ReturnValue;
							}
							else
							{
								Items = null;
							}
							IsRefreshing = false;
						}
						returnValue = null;
						storageDomainId = Guid.Empty;
						break;
				}
			}
		}
	}
}
