/**
 * Copyright 2018 The original authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.dekorate.testing.knative;

import static java.util.Arrays.stream;

import java.lang.reflect.Field;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Optional;

import org.junit.jupiter.api.extension.ExtensionContext;
import org.junit.jupiter.api.extension.TestInstancePostProcessor;

import io.dekorate.DekorateException;
import io.dekorate.testing.WithBaseConfig;
import io.dekorate.testing.WithClosables;
import io.dekorate.testing.WithDiagnostics;
import io.dekorate.testing.annotation.Named;
import io.fabric8.knative.client.KnativeClient;
import io.fabric8.knative.serving.v1.Route;

/**
 * Mixin for storing / loading the Knative Routes to context.
 * It also provides methods for injecting the list.
 */
public interface WithKnativeRoute
    extends TestInstancePostProcessor, WithBaseConfig, WithKnativeClient, WithClosables, WithDiagnostics {

  default void postProcessTestInstance(Object testInstance, ExtensionContext context) throws Exception {
    stream(testInstance.getClass().getDeclaredFields())
        .forEach(f -> injectKnativeRoute(context, testInstance, f));
  }

  /**
   * Inject a {@link Route} to the specified {@link Field}.
   * The pod is matched using its corresponding endpoints.
   * In other words this acts like `inject pod of service`
   * 
   * @param context The execution context.
   * @param testInstance The target test instance.
   * @param field The field to inject.
   */
  default void injectKnativeRoute(ExtensionContext context, Object testInstance, Field field) {
    if (!field.getType().isAssignableFrom(Route.class) && !field.getType().isAssignableFrom(URL.class)) {
      return;
    }

    //This is to make sure we don't write on fields by accident.
    //Note: we don't require the exact annotation. Any annotation named Inject will do (be it javax, guice etc)
    if (!stream(field.getDeclaredAnnotations()).filter(a -> a.annotationType().getSimpleName().equalsIgnoreCase("Inject"))
        .findAny().isPresent()) {
      return;
    }

    String name = namedAnnotationForKnativeRoute(field).orElseGet(() -> getName(context));
    Route route = routeForName(context, name);
    field.setAccessible(true);
    try {
      if (field.getType().isAssignableFrom(Route.class)) {
        field.set(testInstance, route);
      } else if (field.getType().isAssignableFrom(URL.class)) {
        field.set(testInstance, new URL(route.getStatus().getUrl()));
      }
    } catch (IllegalAccessException | MalformedURLException e) {
      throw DekorateException.launderThrowable(e);
    }
  }

  default Route routeForName(ExtensionContext context, String service) {
    KnativeClient client = getKnativeClient(context);
    return client.routes().withName(service).get();
  }

  /**
   * Returns the value of the {@link Named} annotation.
   *
   * @param field The target field.
   * @return An optional string with the name if the field is annotated or empty otherwise.
   */
  default Optional<String> namedAnnotationForKnativeRoute(Field field) {
    return stream(field.getDeclaredAnnotations())
        .filter(a -> a.annotationType().isAssignableFrom(Named.class))
        .map(a -> field.getAnnotation(Named.class).value())
        .findFirst();
  }

  /**
   * @param context The execution context.
   * @return the resource name.
   */
  String getName(ExtensionContext context);

}
